// C Source File
// Created 22/06/2006; 14:57:56

#include "tigcclib.h"
#include "gfabasic.h"
#include "memory.h"
#include "files.h"
#include "charset.h"
#include "utilities.h"

static SYM_ENTRY *GFA_File_TableOpen[GFA_FILE_MAX_OPEN];
static short GFA_File_TableOpen_Size;

static short GFA_File_FindFromAddress_(const GFA_File *file) {
  for (short i = GFA_File_TableOpen_Size - 1; i>=0 ; --i)
    if (GFA_File_TableOpen[i]==file->sym_entry)
      return i;
  return -1;
}


/**
 * @brief Retourne le chemin complet d'un fichier  partir de son nom
 *
 * @param[in] file : Nom de fichier
 * @return un chemin complet vers le fichier, sinon NULL
 *
 * @note Fonction en assembleur
 */
const char *ASM_File_Find(register const char *file asm("%a0")) __attribute__((__regparm__));


GFA_ERROR_NUMBER GFA_File_Open(GFA_File *file, const char *name) {
  Assert(file && name);
  
  static char FileSymStr_buff[1 + 8 + 1 + 8 + 1];
  HSym entry_hsym;
   
  if (!CheckFileName(name))
    return GFA_ERR_INCORRECT_FILE_NAME;
        
  file->size = 0;
  file->data = NULL;
  file->ptr = NULL;
  file->handle = H_NULL;
  
  entry_hsym = SymFind(MakeSymStr(name, FileSymStr_buff, false));
  if ((entry_hsym.folder==H_NULL) && (get_file_name(name)==name)) {
    const char *pname = ASM_File_Find(name);
    if (pname)
    	entry_hsym = SymFind(MakeSymStr(pname, FileSymStr_buff, false));
  }
    
  if ((entry_hsym.folder==H_NULL) || (!(file->sym_entry = DerefSym(entry_hsym))))
    return GFA_ERR_FILE_NO_FOUND;
    
  if (file->sym_entry->flags.flags_n & (SF_OPEN | SF_BUSY))
    return GFA_ERR_FILE_ALREADY_OPEN; 
  
  if ((file->handle = HeapLock(file->sym_entry->handle))==H_NULL)
    return GFA_ERR_IMPOSSIBLE_TO_OPEN_FILE;  
  
  file->sym_entry->flags.flags_n |= (SF_OPEN | SF_BUSY);
     
  void *ptr = HeapDeref(file->handle);
            
  file->data = (char *)(ptr + 2);
  file->size = *(unsigned short *)(ptr) - 2;
  
  unsigned short ftag = *(unsigned char *)(file->data + file->size + 1);
  
  switch (ftag) {
  	case OTH_TAG:
  	  file->data += 2;
      file->size -= 2;
    break;
    
    case TEXT_TAG:
      file->data += 3;
      file->size -= 3;
    break;
    
    default:
      return GFA_ERR_INCORRECT_FILE_TYPE;
  }
        
  file->ptr = (char *)file->data;
  file->ptr_end = (char *)(file->data + file->size);
  
  // Enregistrement information comme quoi le fichier est ouvert
  short idx = GFA_File_FindFromAddress_(NULL);
  if (idx==-1) {
    Assert(GFA_File_TableOpen_Size < GFA_FILE_MAX_OPEN);
    idx = GFA_File_TableOpen_Size++;
  }
  GFA_File_TableOpen[idx] = file->sym_entry;
      
  return GFA_ERR_OK;
}


void GFA_File_Close(GFA_File *file) {
  Assert(file);
  
  #if _DEBUG
    Assert(HeapUnlock(file->handle));
  #else
    HeapUnlock(file->handle);
  #endif
    
  short idx = GFA_File_FindFromAddress_(file);
  Assert(idx!=-1);
  file->sym_entry->flags.flags_n &= ~(SF_OPEN | SF_BUSY);
  GFA_File_TableOpen[idx] = NULL;
  file->sym_entry = NULL;
}


void GFA_File_CloseAll() {
  for (short i = GFA_File_TableOpen_Size - 1; i>=0 ; --i) {
    if (GFA_File_TableOpen[i]!=NULL) {
    	HeapUnlock(GFA_File_TableOpen[i]->handle);
      GFA_File_TableOpen[i]->flags.flags_n &= ~(SF_OPEN | SF_BUSY);
      GFA_File_TableOpen[i] = NULL;
    }
  }
}


inline bool GFA_File_Eof(const GFA_File *file) {
  Assert(file);
  
  return (file->ptr>=file->ptr_end);
}


/*inline char *GFA_File_GetData(const GFA_File *file) {
	return (file->ptr);
}*/


inline unsigned short GFA_File_GetSize(const GFA_File *file) {
	Assert(file);
	
	return file->size;	
}


inline const char *GFA_File_GetEndPos(const GFA_File *file) {
	Assert(file);
	
	return file->ptr_end;
}


inline unsigned short GFA_File_GetType(const GFA_File *file) {
	Assert(file);
	
	return (unsigned short)*(unsigned char *)(file->ptr_end + 1);
}


GFA_ERROR_NUMBER GFA_File_GenerateFileFromMemoryBuffer(GFA_MemoryBuffer *buffer, const char *name, const char *ext) {
  Assert(buffer && name && buffer->fpos && (strlen(ext)<=4));
    
  static char FolderSymStr_buff[1 + 8 + 1];
  static char FileSymStr_buff[1 + 8 + 1 + 8 + 1];
  SYM_STR FolderSymStr = NULL, 
          FileSymStr = NULL;
  
  char *file_name = get_file_name(name);
  static char FileName[8 + 1 + 8 + 1];
  
  HSym entry_hsym;
  SYM_ENTRY *SymPtr;
  HANDLE handle;
  
  // Nom de fichier incorrect
  if (!CheckFileName(name))
    return GFA_ERR_INCORRECT_FILE_NAME;
  
  // Ecriture de l'extension
  GFA_MemoryBuffer_WriteByte(buffer, CHAR_NULL);
  GFA_MemoryBuffer_WriteData(buffer, ext, strlen(ext) + 1);
  GFA_MemoryBuffer_WriteByte(buffer, OTH_TAG);
  
  // Ecriture taille du fichier
  unsigned short file_size = buffer->last_pos - buffer->fpos - 2;
  *(unsigned short *)(buffer->fpos) = file_size;
  *(unsigned short *)(buffer->fpos + sizeof(unsigned short)) = file_size - 8;
  
  FolderSymStr = MakeSymStr((file_name!=name)?name:GFA_BASIC_WRITE_FOLDER, FolderSymStr_buff, true);
  if (FolderSymStr && 
     (FolderFind(FolderSymStr)>=NOT_FOUND) && (FolderAdd(FolderSymStr)==H_NULL))
    return GFA_ERR_IMPOSSIBLE_TO_CREATE_FILE;
  
  // Nom complet du fichier avec chemin
  strcpy(FileName, FolderSymStr_buff + 1);
  if (*(FolderSymStr_buff + 1))
    strcat(FileName, "\\");
  strcat(FileName, file_name);
  
  // Fichier existant est protg en criture ?
  FileSymStr = MakeSymStr(FileName, FileSymStr_buff, false);
  if (((entry_hsym = SymFind(FileSymStr)).folder) && 
  	  (SymPtr = DerefSym(entry_hsym)) &&
  	  (SymPtr->flags.flags_n & (SF_LOCKED | SF_OPEN | SF_ARCHIVED | SF_BUSY)))
    return GFA_ERR_IMPOSSIBLE_TO_CREATE_FILE;
  
  // Cration du symbol dans la VAT
  FileSymStr = MakeSymStr(file_name, FileSymStr_buff, false);  
  entry_hsym = FolderSymStr?AddSymToFolder(FileSymStr, FolderSymStr):SymAdd(FileSymStr);
  if ((entry_hsym.folder==H_NULL) ||
      !(SymPtr = DerefSym(entry_hsym)) /*||
      (SymPtr->flags.flags_n & (SF_LOCKED | SF_OPEN | SF_ARCHIVED | SF_BUSY))*/)
    return GFA_ERR_IMPOSSIBLE_TO_CREATE_FILE;  
  
  // Attribution du handle
  handle = GFA_Memory_GetHandle(GFA_Memory_Realloc(buffer->fpos, buffer->current_pos - buffer->fpos));
  SymPtr->handle = HeapUnlock(handle);
  GFA_Memory_FreeHandle(handle);
  
  buffer->fpos = buffer->epos = buffer->current_pos = NULL;
  buffer->size = 0;

  return GFA_ERR_OK;
}
