// C Source File
// Created 22/06/2006; 18:15:51

#include "tbl_tags.h"
#include "config.h"
#include "tigcclib.h"
#include "gfabasic.h"
#include "errors.h"
#include "memory.h"
#include "files.h"
#include "expr_find.h"
#include "tags.h"
#include "scanner.h"
#include "tokenisor.h"
#include "ETokenisor.h"
#include "utilities.h"

struct s_Params GFA_BASIC_PARAMETERS;
GFA_StackSet GFA_Default_StackSet;

static INT_HANDLER OldInt1 = NULL;
static INT_HANDLER OldInt2 = NULL;
static INT_HANDLER OldInt5 = NULL;

void GFA_StackSet_Create(GFA_StackSet *set) {
	Assert(set);
	
  // Cration de la pile qui va accueillir l'ensemble des librairies ouvertes
  set->stack_lib = GFA_Lib_CreateStack();
  
	// Cration de la pile qui va recevoir les tags
  set->stack_tag = GFA_Tag_CreateStack();
  
  // Cration de la pile qui va recevoir l'ensemble des noms de variables, labels et procdures
  set->stack_expr_find = GFA_ExprFind_CreateStack();
}


void GFA_StackSet_Delete(GFA_StackSet *set) {
	Assert(set);
	
	GFA_Lib_DeleteStack(&set->stack_lib);
  GFA_Tag_DeleteStack(&set->stack_tag);
  GFA_ExprFind_DeleteStack(&set->stack_expr_find);	
}


void GFA_Init() {
  // Initialisation des paramtres
  GFA_BASIC_PARAMETERS.ASSIGN_CURSOR_ERROR = true;
  GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VARIABLE_TYPE = GFA_SUBTAG_FLOAT;
  GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VALUE_TYPE = GFA_SUBTAG_INT_S32;
  
  // Dsactive messages d'erreurs de AMS et optimise en vitesse le programme
  OldInt1 = GetIntVec(AUTO_INT_1);
  OldInt2 = GetIntVec(AUTO_INT_2);
  OldInt5 = GetIntVec(AUTO_INT_5);
  SetIntVec(AUTO_INT_1, DUMMY_HANDLER);
  SetIntVec(AUTO_INT_2, DUMMY_HANDLER);
  SetIntVec(AUTO_INT_5, DUMMY_HANDLER);
  
  // Permet de librer correctement les ressources lors d'une exception
  GFA_Error_Push(GFA_Uninit);
  
  // Allocation du jeu de piles
  GFA_StackSet_Create(&GFA_Default_StackSet);
}


void GFA_Uninit() {
  GFA_StackSet_Delete(&GFA_Default_StackSet);

  SetIntVec(AUTO_INT_5, OldInt5);
  SetIntVec(AUTO_INT_2, OldInt2);
  SetIntVec(AUTO_INT_1, OldInt1);
}


GFA_Error GFA_ParseSourceFile(GFA_File *source_file, const char *dst, GFA_StackSet *set) {
	Assert(source_file && set);
	
	GFA_Error err = GFA_GET_ERR_OK;
	short line_count = 0;
	
	// Ce n'est pas un fichier texte
	if (GFA_File_GetType(source_file)!=TEXT_TAG)
	  return GFA_GET_ERR_N(GFA_ERR_INCORRECT_FILE_TYPE, NULL);
	
	if (!CheckFileName(dst))
	  return GFA_GET_ERR_N(GFA_ERR_INCORRECT_FILE_NAME, NULL);
	
	GFA_ETokensBuffer tokens_buffer = GFA_ETokenisor_CreateBuffer();
	
	// Parcours le fichier source
	while (!GFA_File_Eof(source_file)) {
  	GFA_Error_SetMark((char *)source_file->data, ++line_count, source_file->ptr);
  	
  	err = GFA_Scanner_Process(source_file->ptr, (const char **)&source_file->ptr,
  	                          &set->stack_lib, &set->stack_expr_find, &set->stack_tag);
    
    if (err.n)
      goto end_function;
    
    // Avance de 2 octets, se place sur une nouvelle ligne
  	source_file->ptr += 2;
  	  	
  	// Parsing
  	err = GFA_Parser_Process(&set->stack_lib, &set->stack_expr_find, &set->stack_tag);
  	if (err.n)
  	  goto end_function;
	
	  err.n = GFA_ETokenisor_WriteStackTag(&tokens_buffer, &set->stack_tag);
	  if (err.n)
	    goto end_function; 
	   
    #ifdef _DEBUG
      GFA_Tag_PrintStack((const GFA_StackLib *)&set->stack_lib, &set->stack_tag);
    #endif
      
    // Efface pile de tags
    GFA_Tag_ClearStack(&set->stack_tag);
  }
  
  if (!GFA_ETokenisor_CreateFile(&tokens_buffer, dst, &set->stack_expr_find))
    return GFA_GET_ERR_OK;
  
  end_function:
    GFA_ETokenisor_DeleteBuffer(&tokens_buffer);
  
    return err;
}
