// C Source File
// Created 03/07/2006; 11:55:57

#include "tigcclib.h"
#include "gfalib.h"
#include "gfabasic.h"
#include "errors.h"
#include "memory.h"
#include "strings.h"
#include "files.h"
#include "charset.h"
#include "utilities.h"

const char GFA_LIB_HEADER_SIGNATURE[6] = "GFALib";


/*******************************************************
                 ACCESSEURS MEMBRES PUBLICS
 *******************************************************/
inline unsigned char GFA_Lib_Object_GetTokenIndex(const GFA_Lib_Object *obj) {
	Assert(obj);
	
	return obj->idx_token;
}


inline GFA_enumTypeFunction GFA_Lib_Object_GetFunctionType(const GFA_Lib_Object *obj) {
	Assert(obj);
	
	return obj->function_type;
}


inline GFA_enumTypeConstant GFA_Lib_Object_GetConstantType(const GFA_Lib_Object *obj) {
	Assert(obj);
	
	return obj->constant_type;
}


/*******************************************************
                      METHODES PRIVEES
 *******************************************************/
static inline GFA_String GFA_Lib_GetConstantName_(const GFA_Lib *lib, unsigned short n) {
  Assert(lib && (n < lib->nfunc));
  
  const char *ptr = (const char *)(lib->file.data + lib->tconstant_name[n]);
  return GFA_String_Create(ptr + 1, *ptr);  
}


static inline GFA_String GFA_Lib_GetFunctionName_(const GFA_Lib *lib, unsigned short n) {
  Assert(lib && (n < lib->nfunc));
  
  const char *ptr = (const char *)(lib->file.data + lib->tfunc_name[n]);
  return GFA_String_Create(ptr + 1, *ptr);  
}


static GFA_enumTypeConstant GFA_Lib_GetTypeOfConstant_(const GFA_Lib *lib, unsigned short n, unsigned short *token_pos) {
	Assert(lib && n < (lib->total_constant));
	
	unsigned short count = 0;
  for (unsigned short i = 0; i < GFA_CONSTANT_NUMBER_OF_TYPE; ++i) {
  	if (n < (count + lib->nconstant[i])) {
  		*token_pos = (i - count);
  	  return i;
  	}
  	
  	count += lib->nconstant[i];
  }
	
	Assert(false);
	return -1;
}


static inline const unsigned char *GFA_Lib_GetFunctionParamsAddress_(const GFA_Lib *lib, unsigned short n) {
	Assert(lib && (n < lib->nfunc));
	
	const unsigned char *ptr = (const unsigned char *)(lib->file.data + lib->tfunc_name[n]);
	return (const unsigned char *)(ptr + *ptr + 1);
}


static inline GFA_enumTypeFunction GFA_Lib_GetTypeOfFunction_(const GFA_Lib *lib, unsigned short n) {
	Assert(lib && n < (lib->nfunc));
	
	return (GFA_enumTypeFunction)*GFA_Lib_GetFunctionParamsAddress_(lib, n);
}


static inline unsigned char GFA_Lib_GetConstantTokenIndex_(const GFA_Lib *lib, unsigned short idx_order) {
	Assert(idx_order < lib->total_constant);
	
	return lib->tconstant_idx_tokens[idx_order];
}


static inline unsigned char GFA_Lib_GetFunctionTokenIndex_(const GFA_Lib *lib, unsigned short idx_order) {
	Assert(idx_order < lib->nfunc);
	
	return lib->tfunc_idx_tokens[idx_order];
}



/*******************************************************
                  METHODES PUBLIQUES
 *******************************************************/
bool GFA_Lib_GetConstantObjectFromToken(GFA_Lib_Object *obj, const GFA_Lib *lib, unsigned short token_idx) {
	Assert(obj && lib && (token_idx < lib->total_constant));
	
	for (unsigned short i = 0; i < lib->total_constant; ++i) {
		if (lib->tconstant_idx_tokens[i]==token_idx) {
			obj->idx_order = i;
			obj->idx_token = token_idx; 
			obj->constant_type = GFA_Lib_GetTypeOfConstant_(lib, i, &obj->token_pos);
			return true;
		}
	}
	
	return false;
}


bool GFA_Lib_GetFunctionObjectFromToken(GFA_Lib_Object *obj, const GFA_Lib *lib, unsigned short token_idx) {
	Assert(obj && lib && (token_idx < lib->nfunc));
		
	for (unsigned short i = 0; i < lib->nfunc; ++i) {
		if (lib->tfunc_idx_tokens[i]==token_idx) {
			obj->idx_order = i;
			obj->idx_token = token_idx;
			obj->token_pos = 0;
			obj->function_type = GFA_Lib_GetTypeOfFunction_(lib, i);
			return true;
		}
	}
	
	return false;
}


inline GFA_String GFA_Lib_GetConstantName(const GFA_Lib *lib, const GFA_Lib_Object *obj) {
	Assert(lib && obj);
	
	return GFA_Lib_GetConstantName_(lib, obj->idx_order);
}


inline GFA_String GFA_Lib_GetFunctionName(const GFA_Lib *lib, const GFA_Lib_Object *obj) {
	Assert(lib && obj);
	
	return GFA_Lib_GetFunctionName_(lib, obj->idx_order);
}


const void *GFA_Lib_GetConstantData(const GFA_Lib *lib, const GFA_Lib_Object *obj) {
	Assert(lib && obj);
	
	switch (obj->constant_type) {
		case GFA_CONSTANT_INTEGER:
		  return (const void *)(lib->tconstant_int_s32[obj->token_pos]);
		 
		case GFA_CONSTANT_FLOAT:
		  return (const void *)&(lib->tconstant_float[obj->token_pos]);
		
		case GFA_CONSTANT_STRING:
		{
			const unsigned char *ptr = (const unsigned char *)(lib->file.data + lib->tconstant_name[obj->idx_order]);
			return (const void *)(ptr + *ptr + 1);
		}
		
		default: {
		  Assert(false);
		  return NULL;
		}
	}
}


bool GFA_Lib_FindConstant(GFA_Lib_Object *obj, const GFA_Lib *lib, const GFA_String *cname) {
  Assert(obj && lib && cname);
	   
  // Algorithme de recherche par dichotomie
  short min = 0;
  short max = lib->total_constant - 1;
  short middle;
  short cmp;

  while (min<=max) {
    middle = (min + max) >> 1;
    
    const GFA_String lname = GFA_Lib_GetConstantName_(lib, middle);
    cmp = GFA_String_strcmpUpperWithX(&lname, cname);
       
    if (!cmp) {
    	obj->idx_order = middle;
    	obj->idx_token = GFA_Lib_GetConstantTokenIndex_(lib, middle);
    	obj->constant_type = GFA_Lib_GetTypeOfConstant_(lib, middle, &obj->token_pos);
      return true;
    }
    
    if (cmp < 0)
      min = middle + 1;
    else
      max = middle - 1;
  }
  
  return false; 
}


bool GFA_Lib_FindFunction(GFA_Lib_Object *obj, const GFA_Lib *lib, const GFA_String *fname) {
	Assert(obj && lib && fname);
	
	// Premire lettre, optimise recherche par la table d'interpolation
	char FirstLetter = GFA_Charset_LetterUpper(*GFA_String_GetStr(fname));
    
  // Algorithme de recherche par dichotomie
  short min = lib->tfunc_idxFirstLetter[FirstLetter - 'A'];
  short max = lib->tfunc_idxFirstLetter[FirstLetter - 'A' + 1] - 1;
  short middle;
  short cmp;

  while (min<=max) {
    middle = (min + max) >> 1;
    
    const GFA_String lname = GFA_Lib_GetFunctionName_(lib, middle);
    cmp = GFA_String_strcmpUpperWithX(&lname, fname);
       
    if (!cmp) {
    	obj->idx_order = middle;
    	obj->idx_token = GFA_Lib_GetFunctionTokenIndex_(lib, middle);
    	obj->token_pos = 0;
    	obj->function_type = GFA_Lib_GetTypeOfFunction_(lib, middle);
      return true;
    }
    
    if (cmp < 0)
      min = middle + 1;
    else
      max = middle - 1;
  }
  
  return false;
}


GFA_Lib_ParamsInfo GFA_Lib_GetParamsInfoOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj) {
	Assert(lib && obj && (obj->idx_order < lib->nfunc));
	
	GFA_Lib_ParamsInfo pinfo;
	unsigned char dta = *(GFA_Lib_GetFunctionParamsAddress_(lib, obj->idx_order) + 1) >> 4;
	pinfo.nlist = dta & 0b0111;
	pinfo.is_repeated = dta & 0b1000;
	return pinfo;
}


unsigned short GFA_Lib_GetNumberParamsOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj, unsigned short list_n) {
	Assert(lib && (obj->idx_order < lib->nfunc));
	
	const unsigned char *ptr = (GFA_Lib_GetFunctionParamsAddress_(lib, obj->idx_order) + 1);
	
	// list_n correcte ?
	Assert(list_n < ((*ptr >> 4) & 0b0111));
		
	unsigned char value = *(ptr + ((list_n + 1) >> 1));
	return (unsigned short)((list_n & 1)?(value>>4):(value & 0b1111));
}


const GFA_Lib_Param *GFA_Lib_GetParamsOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj, short list_n) {
	Assert(lib && (obj->idx_order < lib->nfunc));
	
	const unsigned char *ptr = (GFA_Lib_GetFunctionParamsAddress_(lib, obj->idx_order) + 1);
	short nlist = (short)((*(ptr++) >> 4) & 0b0111);
	
	// list_n correcte ?
	Assert(list_n < nlist);
	
	// Ce place au dbut liste des paramtres
	ptr += (nlist >> 1);
	while (--list_n>=0)
		ptr += (GFA_Lib_GetNumberParamsOfFunction(lib, obj, list_n) * sizeof(GFA_Lib_Param));
		
	return (const GFA_Lib_Param *)ptr;
}


GFA_StackLib GFA_Lib_CreateStack() {
  GFA_StackLib stack;
  stack.item = GFA_Memory_Alloc(GFA_LIB_STACK_SIZE * sizeof(GFA_Lib));
  stack.size = 0;
  return stack;
}


void GFA_Lib_DeleteStack(GFA_StackLib *stack) {
  Assert(stack);
  
  for (short i = stack->size - 1; i>=0; --i)
    GFA_Lib_Close(&stack->item[i]);
  GFA_Memory_Free(stack->item);
}


bool GFA_Lib_PushLib(GFA_StackLib *stack, const GFA_Lib *lib) {
  Assert(stack && lib);
  
  if (stack->size>=GFA_LIB_STACK_SIZE)
    return false;
  
  stack->item[stack->size++] = *lib;  
  return true;
}


inline const GFA_Lib *GFA_Lib_GetLib(const GFA_StackLib *stack, unsigned short n) {
  Assert(stack && (n < (stack->size)));
  return &stack->item[n];
}


short GFA_Lib_FindConstantInStack(GFA_Lib_Object *obj, const GFA_StackLib *stack, const GFA_String *cname) {
  Assert(obj && stack && cname);
  
  for (short i = (stack->size - 1); i>=0; --i) {
    if (GFA_Lib_FindConstant(obj, GFA_Lib_GetLib(stack, i), cname))
      return i;
  }
  
  return -1;
}


short GFA_Lib_FindFunctionInStack(GFA_Lib_Object *obj, const GFA_StackLib *stack, const GFA_String *fname) {
  Assert(obj && stack && fname);
  
  for (short i = (stack->size - 1); i>=0; --i) {
    if (GFA_Lib_FindFunction(obj, GFA_Lib_GetLib(stack, i), fname))
    	return i;
  }
  
  return -1;
}


GFA_ERROR_NUMBER GFA_Lib_Open(GFA_Lib *lib, const char *name) {  
  Assert(lib && name);
  
  static char buffer[8 + 8 + 1 + 1];
  const char *pname = get_file_name(name);
  GFA_ERROR_NUMBER err;
  
  // Vrification nom du fichier
  if (!CheckFileName(name))
    return GFA_ERR_INCORRECT_FILE_NAME;
   
  // Ouverture du fichier
  if (pname==name) {
    memcpy(buffer, GFA_BASIC_DEFAULT_FOLDER, sizeof(GFA_BASIC_DEFAULT_FOLDER));
    buffer[sizeof(GFA_BASIC_DEFAULT_FOLDER) - 1] = '\\';
    strcpy((char *)(buffer + sizeof(GFA_BASIC_DEFAULT_FOLDER)), name);
    err = GFA_File_Open(&lib->file, buffer);
    if (err==GFA_ERR_FILE_NO_FOUND)
      err = GFA_File_Open(&lib->file, name);
    if (err)
      return err;
  }

  // Enregistrement du nom en majuscules
  StrUpper(lib->name, pname, strlen(pname));

  // Extension incorrecte
  if (GFA_File_GetType(&lib->file)!=OTH_TAG) {
	  err = GFA_ERR_INCORRECT_FILE_TYPE;
	  goto end_error;
	}

  // Lecture signature du fichier
  if (memcmp(GFA_LIB_HEADER_SIGNATURE, lib->file.data, sizeof(GFA_LIB_HEADER_SIGNATURE))) {
  	err = GFA_ERR_INCORRECT_FILE_SIGNATURE;
  	goto end_error;
  }
  char *ptr = (char *)lib->file.data + sizeof(GFA_LIB_HEADER_SIGNATURE);
  
  // Lecture version
  if (!CheckVersion(GFA_LIB_HEADER_FILE_VERSION_REQUIRE, *((unsigned short *)(ptr)++))) {
  	err = GFA_ERR_INCORRECT_FILE_VERSION;
  	goto end_error;
  }
  
  // Version de la librairie
  lib->version_major = *(ptr++);
  lib->version_minor = *(ptr++);
  
  // Saute offset dbut table d'allocations
  ptr += sizeof(unsigned short);
  
  // Nombre de constantes
  lib->nconstant = (unsigned char *)ptr;
  ptr += GFA_CONSTANT_NUMBER_OF_TYPE * sizeof(unsigned char);
  
  lib->total_constant = 0;
  for (unsigned short i = 0; i < GFA_CONSTANT_NUMBER_OF_TYPE; ++i)
    lib->total_constant += lib->nconstant[i];
  
  // Nombre de fonctions
  lib->nfunc = *(unsigned char *)ptr++;
  
  // Table d'offsets, noms des constantes
  lib->tconstant_name = (unsigned short *)(ptr);
  ptr += lib->total_constant * sizeof(unsigned short);
  
  // Table d'offsets, noms des fonctions
  lib->tfunc_name = (unsigned short *)(ptr);
  ptr += lib->nfunc * sizeof(unsigned short);
  
  // Correspondance idx name vers idx token
  lib->tconstant_idx_tokens = ptr;
  ptr += lib->total_constant * sizeof(unsigned char);
  
  // Correspondance idx name vers idx token
  lib->tfunc_idx_tokens = ptr;
  ptr += lib->nfunc * sizeof(unsigned char);
  
  // Alignement si ncessaire
  if ((unsigned long)ptr & 1UL)
    ++ptr;
  
  // Donnes des constantes (integer)
  lib->tconstant_int_s32 = (const long *)ptr;
  ptr += lib->nconstant[GFA_CONSTANT_INTEGER];
  
  // Donnes des constantes (float)
  lib->tconstant_float = (const float *)ptr;
  ptr += lib->nconstant[GFA_CONSTANT_FLOAT];
  
  // Cration de la table d'interpolation
  unsigned char letter = 'A';
  unsigned short start = lib->nfunc?0:lib->nfunc;
  for (unsigned short i = 0; i < lib->nfunc; ++i) {
    char c = *(const char *)(lib->file.data + lib->tfunc_name[i] + 1);
      
    if (c!=letter) {
      lib->tfunc_idxFirstLetter[letter - 'A'] = start;
      while (c!=++letter)
        lib->tfunc_idxFirstLetter[letter - 'A'] = i;
      start = i;
    }
  }
  lib->tfunc_idxFirstLetter[letter - 'A'] = start;
  while (++letter<='`')
    lib->tfunc_idxFirstLetter[letter - 'A'] = lib->nfunc;
  
  return GFA_ERR_OK;
  
  // Une erreur s'est produite
  end_error:
    GFA_File_Close(&lib->file);
    return err;
}


inline void GFA_Lib_Close(GFA_Lib *lib) {
  Assert(lib);
  
  GFA_File_Close(&lib->file);
}
