// Header File
// Created 03/07/2006; 11:55:47

#ifndef _GFA_LIB_H_
#define _GFA_LIB_H_

#include "errors.h"
#include "strings.h"
#include "files.h"

#define GFA_LIB_STACK_SIZE              16    // Nombre total de bibliothques que l'on peut inclure

/** @brief Signature d'une librairie */
extern const char GFA_LIB_HEADER_SIGNATURE[6];

/** @brief Version supporte pour les librairies */
#define GFA_LIB_HEADER_FILE_VERSION_REQUIRE   0x0102

/** @brief Structure d'une librairie */
typedef struct {
  char name[8 + 1];
  unsigned char nfunc;
  unsigned short tfunc_idxFirstLetter[('`' - 'A') + 1];
  GFA_File file;
  unsigned char version_major, version_minor;
  const unsigned char *nconstant;
  unsigned short total_constant;
  const unsigned short *tconstant_name;
  const unsigned short *tfunc_name;
  const unsigned char *tconstant_idx_tokens;
  const unsigned char *tfunc_idx_tokens;
  const long *tconstant_int_s32;
  const float *tconstant_float;
} GFA_Lib;


/** @brief Structure pour enregistrer une bibliothque de librairies */
typedef struct {
  GFA_Lib *item;
  unsigned short size;
} GFA_StackLib;


/** @brief Enumration des types de constantes disponibles */
typedef enum {
  GFA_CONSTANT_INTEGER,
  GFA_CONSTANT_FLOAT,
  GFA_CONSTANT_STRING,
  
  GFA_CONSTANT_NUMBER_OF_TYPE = 3  
} GFA_enumTypeConstant;


typedef union {
	struct {
	  unsigned char STRING : 1, BOOLEAN : 1, FLOAT : 1, S32 : 1, S16 : 1, U8 : 1;
	};
	unsigned char bit_field;
} GFA_enumTypeFunction;


enum {
	RETURN_U8      = 0x01,
	RETURN_S16     = 0x02,
	RETURN_S32     = 0x04,
	RETURN_FLOAT   = 0x08,
	RETURN_BOOLEAN = 0x10,
	RETURN_STRING  = 0x20
} GFA_enumTypeFunctionFlags;


#define GFA_LIB_FUNC_RETURN_FIRST_PARAM_TYPE(x)    (x.bit_field==0xFF)
#define GFA_LIB_FUNC_RETURN_VALUE(x)               (x.bit_field)                   

/** @brief Structure renvoye lors d'une recherche de fonction ou de constante */
typedef struct {
	unsigned char idx_order;
	unsigned char idx_token;
	short token_pos;  // Position du n de token dans la liste
	union {
    GFA_enumTypeFunction function_type; // Valide uniquement pour les fonctions
    GFA_enumTypeConstant constant_type; // Valide uniquement pour les constantes
  };
} GFA_Lib_Object;


typedef struct {
	unsigned short nlist;
	bool is_repeated;
} GFA_Lib_ParamsInfo;


typedef struct {
  union {
		struct {
      unsigned char FACULT : 1, NOT_ATTRIB : 4, VARTABLE : 1, VAR : 1, VALUE : 1;
    };
	  unsigned char bit_field;
  } container;
		
	union { 
	  struct {
	    unsigned char NOT_ATTRIB : 1, CHANNEL : 1, STRING : 1, BOOLEAN : 1, FLOAT : 1, S32 : 1, S16 : 1, U8 : 1;
	  };
	  unsigned char bit_field;
	} type;
} GFA_Lib_Param;


enum GFA_enumContainerFlags {
	CONTAINER_VALUE    = 0x01,
	CONTAINER_VAR      = 0x02,
	CONTAINER_VARTABLE = 0x04,
	CONTAINER_FACULT   = 0x80
};


enum GFA_enumTypeFlags {
	TYPE_U8      = 0x01,
	TYPE_S16     = 0x02,
	TYPE_S32     = 0x04,
	TYPE_FLOAT   = 0x08,
	TYPE_BOOLEAN = 0x10,
	TYPE_STRING  = 0x20,
	TYPE_CHANNEL = 0x40
};

#define GFA_LIB_PARAM_FACULT(x)								(x.container.bit_field & CONTAINER_FACULT)
#define GFA_LIB_PARAM_PREVIEW_CONTAINER(x)    (!(x.bit_field & (~CONTAINER_FACULT)))
#define GFA_LIB_PARAM_PREVIEW_TYPE(x)					(!x.bit_field)


/*******************************************************
                 ACCESSEURS MEMBRES PUBLICS
 *******************************************************/
extern inline unsigned char GFA_Lib_Object_GetTokenIndex(const GFA_Lib_Object *obj);
extern inline GFA_enumTypeFunction GFA_Lib_Object_GetFunctionType(const GFA_Lib_Object *obj);
extern inline GFA_enumTypeConstant GFA_Lib_Object_GetConstantType(const GFA_Lib_Object *obj);


/*******************************************************
                  METHODES PUBLIQUES
 *******************************************************/
extern bool GFA_Lib_GetConstantObjectFromToken(GFA_Lib_Object *obj, const GFA_Lib *lib, unsigned short token_idx);
extern bool GFA_Lib_GetFunctionObjectFromToken(GFA_Lib_Object *obj, const GFA_Lib *lib, unsigned short token_idx);
extern inline GFA_String GFA_Lib_GetConstantName(const GFA_Lib *lib, const GFA_Lib_Object *obj);
extern inline GFA_String GFA_Lib_GetFunctionName(const GFA_Lib *lib, const GFA_Lib_Object *obj);
extern const void *GFA_Lib_GetConstantData(const GFA_Lib *lib, const GFA_Lib_Object *obj);
extern bool GFA_Lib_FindConstant(GFA_Lib_Object *obj, const GFA_Lib *lib, const GFA_String *cname);
extern bool GFA_Lib_FindFunction(GFA_Lib_Object *obj, const GFA_Lib *lib, const GFA_String *fname);
extern GFA_Lib_ParamsInfo GFA_Lib_GetParamsInfoOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj);
extern unsigned short GFA_Lib_GetNumberParamsOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj, unsigned short list_n);
extern const GFA_Lib_Param *GFA_Lib_GetParamsOfFunction(const GFA_Lib *lib, const GFA_Lib_Object *obj, short list_n);


/**
 * @brief Cration d'une pile (bibliothque) de librairies
 *
 * @return une bibliothque vide de librairies
 */
extern GFA_StackLib GFA_Lib_CreateStack();


/**
 * @brief Efface une bibliothque de librairies
 *
 * @param[in, out] stack : Pile de bibliothque
 */
extern void GFA_Lib_DeleteStack(GFA_StackLib *stack);


/**
 * @brief Ajoute une librairie dans une bibliothque
 *
 * @param[in, out] stack : Pile de librairies
 * @param[in] lib : Librairie
 * @return true si la librairie a te correctement ajoute, sinon false
 */
extern bool GFA_Lib_PushLib(GFA_StackLib *stack, const GFA_Lib *lib);


/**
 * @brief Retourne une librairie  partir d'une bibliothque
 *
 * @param[in] stack : Bibliothque de librairies
 * @param[in] n : Indice de la librairie dans la bibliothque
 * @return la librairie
 *
 * @note n doit tre un indice valide
 */
extern inline const GFA_Lib *GFA_Lib_GetLib(const GFA_StackLib *stack, unsigned short n);


extern short GFA_Lib_FindConstantInStack(GFA_Lib_Object *obj, const GFA_StackLib *stack, const GFA_String *cname);
extern short GFA_Lib_FindFunctionInStack(GFA_Lib_Object *obj, const GFA_StackLib *stack, const GFA_String *fname);


/**
 * @brief Ouvre un fichier correspondant  une librairie
 *
 * @param[out] lib : Structure accueillant les informations de la librairie  ouvrir
 * @param[in] name : Nom de la librairie
 * @return un code d'erreur
 *
 * @note Si name ne contient pas un nom de dossier la recherche est faite en priorit
 * dans le dossier par dfaut de GFA_BASIC
 */
extern GFA_ERROR_NUMBER GFA_Lib_Open(GFA_Lib *lib, const char *name);


/**
 * @brief Ferme une librairie
 *
 * @param[in] lib : Librairie
 */
extern inline void GFA_Lib_Close(GFA_Lib *lib);

#endif