// C Source File
// Created 25/06/2006; 16:19:34

#include "tigcclib.h"
#include "memory.h"
#include "errors.h"

/** @brief Table d'allocations */
static struct s_Memory_Alloc {
  HANDLE handle;
  void *address;
} GFA_Memory_TableAlloc[GFA_MEMORY_MAX_ALLOC];    

/** @brief Taille de la table d'allocations */
static short GFA_Memory_TableAlloc_Size;

static short GFA_Memory_FindAllocFromAddress_(void *address) { 
  for (short i = GFA_Memory_TableAlloc_Size - 1; i>=0 ; --i)
    if (GFA_Memory_TableAlloc[i].address==address)
      return i;
  return -1;
}


static short GFA_Memory_FindAllocFromHandle_(HANDLE handle) {
  for (short i = GFA_Memory_TableAlloc_Size - 1; i>=0 ; --i)
    if (GFA_Memory_TableAlloc[i].handle==handle)
      return i;
  return -1;
}


void *GFA_Memory_Alloc(unsigned short size) {
  short idx = GFA_Memory_FindAllocFromHandle_(H_NULL);
  if (idx==-1) {
    Assert(GFA_Memory_TableAlloc_Size < GFA_MEMORY_MAX_ALLOC);
    idx = GFA_Memory_TableAlloc_Size++;
  }
  
  if (((GFA_Memory_TableAlloc[idx].handle = HeapAlloc(size))==H_NULL) ||
      (HeapLock(GFA_Memory_TableAlloc[idx].handle)==H_NULL))
    GFA_Error_Call2(GFA_ERR_MEMORY, NULL);
  
  return (GFA_Memory_TableAlloc[idx].address = HeapDeref(GFA_Memory_TableAlloc[idx].handle));
}


void *GFA_Memory_Realloc(void *address, unsigned short new_size) {
  Assert(address);
  
  short idx = GFA_Memory_FindAllocFromAddress_(address);
  Assert(idx!=-1);
  #ifdef _DEBUG
    Assert(HeapUnlock(GFA_Memory_TableAlloc[idx].handle)!=H_NULL);
  #else
    HeapUnlock(GFA_Memory_TableAlloc[idx].handle);
  #endif
  
  if (((GFA_Memory_TableAlloc[idx].handle = HeapRealloc(GFA_Memory_TableAlloc[idx].handle, new_size))==H_NULL) ||
      (HeapLock(GFA_Memory_TableAlloc[idx].handle)==H_NULL))
    GFA_Error_Call2(GFA_ERR_MEMORY, NULL);
  
  return (GFA_Memory_TableAlloc[idx].address = HeapDeref(GFA_Memory_TableAlloc[idx].handle));
}


void GFA_Memory_Free(void *address) {
  Assert(address);
  
  /*if (!address) 
    return; //Assert(address!=NULL);*/
  
  short idx = GFA_Memory_FindAllocFromAddress_(address);
  Assert(idx!=-1);
  HeapFree(GFA_Memory_TableAlloc[idx].handle);
  GFA_Memory_TableAlloc[idx].handle = H_NULL;
  GFA_Memory_TableAlloc[idx].address = NULL;
}


void GFA_Memory_FreeAll() {
  for (short i = GFA_Memory_TableAlloc_Size - 1; i>=0 ; --i) {
    if (GFA_Memory_TableAlloc[i].handle!=H_NULL)
      HeapFree(GFA_Memory_TableAlloc[i].handle);
  }
}


HANDLE GFA_Memory_GetHandle(void *address) {
  Assert(address);
  
  short idx = GFA_Memory_FindAllocFromAddress_(address);
  return (idx!=-1)?GFA_Memory_TableAlloc[idx].handle:H_NULL;
}


void GFA_Memory_FreeHandle(HANDLE handle) {
  Assert(handle!=H_NULL);
  
  short idx = GFA_Memory_FindAllocFromHandle_(handle);
  Assert(idx!=-1);
  #ifdef _DEBUG
    Assert(HeapUnlock(handle)!=H_NULL);
  #else
    HeapUnlock(handle);
  #endif
  
  GFA_Memory_TableAlloc[idx].handle = H_NULL;
  GFA_Memory_TableAlloc[idx].address = NULL;
}


GFA_MemoryBuffer GFA_MemoryBuffer_Create() {
  Assert(GFA_MEMORY_BUFFER_INIT_SIZE > 4);
  GFA_MemoryBuffer buffer;
  
  buffer.fpos = GFA_Memory_Alloc(GFA_MEMORY_BUFFER_INIT_SIZE);
  buffer.epos = buffer.fpos + (buffer.size = GFA_MEMORY_BUFFER_INIT_SIZE);

  // Avance de 4 octets, soit info taille fichier
  buffer.current_pos = buffer.fpos + sizeof(unsigned long);
  buffer.last_pos = buffer.current_pos;
  return buffer;
}


void GFA_MemoryBuffer_Delete(GFA_MemoryBuffer *buffer) {
  Assert(buffer);
  
  if (!buffer->fpos) return;
  GFA_Memory_Free(buffer->fpos);
  buffer->fpos = buffer->epos = buffer->last_pos = buffer->current_pos = NULL;
  buffer->size = 0;
}


void GFA_MemoryBuffer_Extend(GFA_MemoryBuffer *buffer, unsigned short extend) {
  Assert(buffer && buffer->fpos);
  
  if ((buffer->current_pos + extend) <= buffer->epos)
    return;
  
  void *old_fpos = buffer->fpos;
  for (unsigned short new_size = (buffer->current_pos + extend) - old_fpos;
       (buffer->size += GFA_MEMORY_BUFFER_EXTEND_SIZE) < new_size;);
  buffer->fpos = GFA_Memory_Realloc(old_fpos, buffer->size);
  buffer->current_pos = buffer->fpos + (buffer->current_pos - old_fpos);
  buffer->last_pos = buffer->fpos + (buffer->last_pos - old_fpos);
  buffer->epos = buffer->fpos + buffer->size;
}


void GFA_MemoryBuffer_WriteByte(GFA_MemoryBuffer *buffer, unsigned char byte) {
  Assert(buffer);
  
  GFA_MemoryBuffer_Extend(buffer, sizeof(byte));
  *(unsigned char *)(buffer->current_pos++) = byte;
  
  if (buffer->current_pos > buffer->last_pos)
    buffer->last_pos = buffer->current_pos;
}


void GFA_MemoryBuffer_WriteShort(GFA_MemoryBuffer *buffer, short shrt) {
  Assert(buffer);
  
  GFA_MemoryBuffer_Extend(buffer, sizeof(shrt));
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(shrt >> 8);
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(shrt);
  
  if (buffer->current_pos > buffer->last_pos)
    buffer->last_pos = buffer->current_pos;
}


void GFA_MemoryBuffer_WriteLong(GFA_MemoryBuffer *buffer, long lng) {
  Assert(buffer);
  
  GFA_MemoryBuffer_Extend(buffer, sizeof(lng));
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(lng >> 24);
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(lng >> 16);
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(lng >> 8);
  *(unsigned char *)(buffer->current_pos++) = (unsigned char)(lng);
  
  if (buffer->current_pos > buffer->last_pos)
    buffer->last_pos = buffer->current_pos;
}


void GFA_MemoryBuffer_WriteData(GFA_MemoryBuffer *buffer, const void *data, unsigned short size) {
  Assert(buffer && data);
  
  GFA_MemoryBuffer_Extend(buffer, size);
  memcpy(buffer->current_pos, data, size);
  buffer->current_pos += size;
  
  if (buffer->current_pos > buffer->last_pos)
    buffer->last_pos = buffer->current_pos;
}


inline void *GFA_MemoryBuffer_GetBeginAddress(const GFA_MemoryBuffer *buffer) {
  Assert(buffer);
  
  return (buffer->fpos + sizeof(unsigned long));   
}


inline void *GFA_MemoryBuffer_GetCurrentAddress(const GFA_MemoryBuffer *buffer) {
  Assert(buffer);
  
  return buffer->current_pos;
}


void GFA_MemoryBuffer_SetCurrentAddress(GFA_MemoryBuffer *buffer, long offset, GFA_enumOffsetType type) {
	Assert(buffer);
	
	switch (type) {
		case GFA_SEEK_SET:
		  buffer->current_pos = GFA_MemoryBuffer_GetBeginAddress(buffer) + offset;
		break;
		
		case GFA_SEEK_CUR:
		  buffer->current_pos += offset;
		break;
		  
		case GFA_SEEK_END:
		  buffer->current_pos = buffer->last_pos + offset;
    break;
      
    default:
      Assert(false);
	}
	
	Assert(buffer->current_pos>=GFA_MemoryBuffer_GetBeginAddress(buffer));
	if (buffer->current_pos > buffer->last_pos) {
	  GFA_MemoryBuffer_Extend(buffer, buffer->current_pos - buffer->last_pos);
	  buffer->last_pos = buffer->current_pos;
	}
}


inline unsigned short GFA_MemoryBuffer_GetSize(const GFA_MemoryBuffer *buffer) {
	Assert(buffer);
	
	return (unsigned short)(buffer->last_pos - GFA_MemoryBuffer_GetBeginAddress(buffer));
}