// Header File
// Created 25/06/2006; 16:19:02

#ifndef _GFA_MEMORY_H_
#define _GFA_MEMORY_H_

#define GFA_MEMORY_MAX_ALLOC               64
#define GFA_MEMORY_BUFFER_INIT_SIZE        256
#define GFA_MEMORY_BUFFER_EXTEND_SIZE      256

typedef enum {
	GFA_SEEK_SET,
	GFA_SEEK_CUR,
	GFA_SEEK_END
} GFA_enumOffsetType;

/** @brief Structure reprsentant un buffer */
typedef struct {
  void *fpos;
  void *epos;
  void *current_pos;
  void *last_pos;
  unsigned short size;
} GFA_MemoryBuffer;


/**
 * @brief Effectue une allocation mmoire
 *
 * @param[in] size : Taille de la zone mmoire  rserve en octets
 * @return un pointeur vers la zone alloue, sinon NULL en cas de mmoire insuffisante
 *
 * @note Cette fonction garde une trace de l'allocation
 */
extern void *GFA_Memory_Alloc(unsigned short size);


/**
 * @brief Change la taille d'une allocation mmoire
 *
 * @param[in] address : Zone mmoire
 * @param[in] new_size : Nouvelle taille en octets
 * @return un pointeur vers la nouvelle zone mmoire alloue, sinon NULL en cas de mmoire insuffisante
 *
 * @note Les donnes de la zone mmoire  rallouer sont stocker dans la nouvelle allocation
 */
extern void *GFA_Memory_Realloc(void *address, unsigned short new_size);


/**
 * @brief Dsalloue une zone mmoire
 *
 * @param[in] data : Pointeur vers une zone mmoire alloue par GFA_Memory_Alloc
 */
extern void GFA_Memory_Free(void *address);


/**
 * @brief Efface toutes les allocations mmoire restantes effectues par GFA_Memory_Alloc
 */
extern void GFA_Memory_FreeAll();


/**
 * @brief Retourne le numro d'handle d'une zone mmoire
 *
 * @param[in] address : Zone mmoire alloue
 * @return un numro d'handle
 */
extern HANDLE GFA_Memory_GetHandle(void *address);


/**
 * @brief Libre une zone mmoire du gestionnaire
 *
 * @param[in] handle : Numro d'handle
 *
 * @note Cette fonction unlock l'handle et le libre du gestionnaire mmoire
 * ainsi l'espace l'allocation est toujours prsente en mmoire mais elle
 * n'est plus connue du gestionnaire de mmoire.
 * Cette fonction peut tre utile pour grer une zone mmoire qui deviendra un
 * fichier.
 */
extern void GFA_Memory_FreeHandle(HANDLE handle);


/**
 * @brief Cration d'un buffer relatif  un futur fichier
 *
 * @return un buffer vierge
 *
 * @note En cas de mmoire insuffisante, une exception est excute.
 */
extern GFA_MemoryBuffer GFA_MemoryBuffer_Create();


/**
 * @brief Efface un buffer de fichier
 *
 * @param[in, out] buffer : Buffer
 */
extern void GFA_MemoryBuffer_Delete(GFA_MemoryBuffer *buffer);


/**
 * @brief Augmente si ncessaire un buffer
 *
 * @param[in, out] buffer : Buffer
 * @param[in] extend : Taille de la futur donne  crire
 *
 * @note extend correspond  la taille de la futur donne  crire, 
 * si cette taille excde la taille totale du buffer, le buffer est augment.
 * En cas de mmoire insuffisante, une exception est excute.
 */
extern void GFA_MemoryBuffer_Extend(GFA_MemoryBuffer *buffer, unsigned short extend);


/**
 * @brief Ecrit un octet dans un buffer
 *
 * @param[in, out] buffer :  Buffer
 * @param[in] byte : Octet  crire
 */
extern void GFA_MemoryBuffer_WriteByte(GFA_MemoryBuffer *buffer, unsigned char byte);


/**
 * @brief Ecrit un mot dans un buffer
 *
 * @param[in, out] buffer : Buffer
 * @param[in] shrt : Mot  crire
 */
extern void GFA_MemoryBuffer_WriteShort(GFA_MemoryBuffer *buffer, short shrt);


/**
 * @brief Ecrit un double mot dans un buffer
 *
 * @param[in, out] buffer : Buffer
 * @param[in] lng : Double mot  crire
 */
extern void GFA_MemoryBuffer_WriteLong(GFA_MemoryBuffer *buffer, long lng);


/**
 * @brief Ecrit une donne dans un buffer
 *
 * @param[in, out] buffer : Buffer
 * @param[in] data : Pointeur vers la donne  crire
 * @param[in] size : Nombre d'octets  crire
 */
extern void GFA_MemoryBuffer_WriteData(GFA_MemoryBuffer *buffer, const void *data, unsigned short size);


/**
 * @brief Retourne l'adresse de dbut d'un buffer
 *
 * @param[in] buffer : Buffer
 * @return l'adresse de dbut du buffer
 */
extern inline void *GFA_MemoryBuffer_GetBeginAddress(const GFA_MemoryBuffer *buffer);


/**
 * @brief Retourne l'adresse de fin d'un buffer
 *
 * @param[in] buffer : Buffer
 * @return l'adresse de fin du buffer
 *
 * @note L'adresse retourne correspond  l'adresse o sera crite 
 * la prochaine donne dans le buffer
 */
extern inline void *GFA_MemoryBuffer_GetCurrentAddress(const GFA_MemoryBuffer *buffer);
extern void GFA_MemoryBuffer_SetCurrentAddress(GFA_MemoryBuffer *buffer, long offset, GFA_enumOffsetType type);
extern inline unsigned short GFA_MemoryBuffer_GetSize(const GFA_MemoryBuffer *buffer);

#endif
