// C Source File
// Created 22/06/2006; 18:58:29

#include "tbl_tags.h"
#include "tbl_parser.h"
#include "tbl_structs.h"
#include "config.h"
#include "tigcclib.h"
#include "gfabasic.h"
#include "gfalib.h"
#include "errors.h"
#include "memory.h"
#include "strings.h"
#include "charset.h"
#include "exprs.h"
#include "tags.h"
#include "scanner.h"

static GFA_Error GFA_Scanner_AdjustStack(GFA_StackTag *stack_tag, GFA_StackLib *stack_lib, GFA_StackExprFind *stack_expr_find);
static GFA_Error GFA_Scanner_MakeNumber(GFA_StackTag *stack_tag);
static GFA_Error GFA_Scanner_DecryptParserCommands(GFA_StackLib *stack_lib, GFA_StackTag *stack_tag);
static void GFA_Scanner_FinalizeStack(GFA_StackTag *stack_tag);
static GFA_Error GFA_Scanner_CheckSyntax(GFA_StackTag *stack_tag);

// Dtermine le type des nombres suffixs
static const char *GFA_Scanner_NormalizeNumber(GFA_Tag *tag, const char *ptr) {  
  for (short i = (sizeof(GFA_Number_Suffix_Table) / sizeof(struct s_Number_Suffix_Table)) - 1; i>=0; --i) {
    if (*ptr==GFA_Number_Suffix_Table[i].suffix) {
      tag->subtag = GFA_Number_Suffix_Table[i].subtag;
      return (ptr + 1);
    }
  }
  
  return ptr;
}


// Suffixe aprs un symbole
static const char *GFA_Scanner_NormalizeSymbol(GFA_Tag *tag, const char *ptr) {
  for (short i = (sizeof(GFA_Symbol_Suffix_Table) / sizeof(struct s_Symbol_Suffix_Table)) - 1; i>=0; --i) {
    if (*ptr==GFA_Symbol_Suffix_Table[i].suffix) {
      tag->tag = GFA_Symbol_Suffix_Table[i].tag;
      tag->subtag = GFA_Symbol_Suffix_Table[i].subtag;
      ++tag->expr.size;
      return (ptr + 1);
    }
  }
  
  return ptr;
}


// Signes valides du langage
static const char *GFA_Scanner_NormalizeSign(GFA_Tag *tag, const char *ptr) {
  const char *sav_ptr = ptr;
  for (unsigned short i = 0; i < (sizeof(GFA_Sign_Table) / sizeof(struct s_Sign_Table)); ++i) {
    unsigned short j;
    // Compare le signe courant
    for (j = 0; j < GFA_Sign_Table[i].size; ++j) {
      if (*(ptr++)!=GFA_Sign_Table[i].sign[j])
        break;
    }
    
    // Signe trouv
    if (j==GFA_Sign_Table[i].size) {
      tag->expr = GFA_String_Create(sav_ptr, GFA_Sign_Table[i].size);
      tag->tag = GFA_Sign_Table[i].tag;
      tag->subtag = GFA_Sign_Table[i].subtag;
      return ptr;  
    }
    
    ptr = sav_ptr;
  }
  
  return sav_ptr;
}


GFA_Error GFA_Scanner_Process(const char *buffer, const char **rbuffer,
                              GFA_StackLib *stack_lib, GFA_StackExprFind *stack_expr_find, GFA_StackTag *stack_tag) 
{
  static const struct s_Expr {
    const char *(*func)(const char *);
    GFA_TAG tag;
    GFA_SUBTAG subtag;
  } TblExpr[] = {
    {GFA_Expr_Symbol,       GFA_TAG_SYMBOL, GFA_SUBTAG_NONE},
    {GFA_Expr_Number,       GFA_TAG_VALUE,  GFA_SUBTAG_NONE},
    {GFA_Expr_Number_Float, GFA_TAG_VALUE,  GFA_SUBTAG_FLOAT},
    {GFA_Expr_String,       GFA_TAG_VALUE,  GFA_SUBTAG_STRING}
  };
  
  GFA_Error err;
  GFA_Tag tag;
  const char *line_begin = buffer;	
	const char *process_ptr;
	bool overflow = false;
		
	// Parcours le buffer
	while (!GFA_Charset_EofLine(*buffer)) {
		// Caractre espace
    if (*buffer==' ') {
      ++buffer;
      continue;
    }
    
    // Crer un nouveau tag
    tag = GFA_Tag_Create(NULL, GFA_TAG_UNKNOWN, GFA_SUBTAG_NONE, GFA_PRIORITY_NONE);
    
    // Recherche une expression complexe
    for (short i = (sizeof(TblExpr) / sizeof(struct s_Expr)) - 1; i>=0; --i) {
      if ((process_ptr = (TblExpr[i].func)(buffer))) {
        tag.tag = TblExpr[i].tag;
        if ((tag.tag==GFA_TAG_VALUE) && (TblExpr[i].subtag==GFA_SUBTAG_NONE))
          tag.subtag = GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VALUE_TYPE;
        else
          tag.subtag = TblExpr[i].subtag;
        
        // Retire "..." pour une chane de caractres
        if (tag.symbol==GFA_SYMBOL_VALUE_STRING)
          tag.expr = GFA_String_Create(buffer + 1, (process_ptr - buffer - 2));
        // Cration de l'expression
        else
          tag.expr = GFA_String_Create(buffer, (process_ptr - buffer));
        
        buffer = process_ptr;
        break;
      }    
    }
    
    // Nombre sous forme d'une base (binaire, hexa, octale)
    process_ptr = GFA_Expr_Number_Base(buffer, &tag.info.number.value_S32, &overflow);
    if (overflow) 
      return GFA_GET_ERR(GFA_ERR_NUMBER_OVERFLOW, NULL, buffer);
    if (process_ptr) {
      tag.expr = GFA_String_Create(buffer, (process_ptr - buffer));
      tag.tag = GFA_TAG_BASE;
      tag.subtag = GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VALUE_TYPE;
      buffer = process_ptr;
    }
    
    // Normalisation des nombres
    if (((tag.tag==GFA_TAG_VALUE) || (tag.tag==GFA_TAG_BASE)) && (tag.subtag!=GFA_SUBTAG_STRING))
      buffer = GFA_Scanner_NormalizeNumber(&tag, buffer);
    // Normalisation des symboles
    else if (tag.tag==GFA_TAG_SYMBOL) {
      buffer = GFA_Scanner_NormalizeSymbol(&tag, buffer);

      // Recherche de commentaire type REM
      if (!GFA_Tag_GetStackSize(stack_tag) && 
         (tag.subtag==GFA_SUBTAG_NONE) && !(GFA_String_StrcmpUpper(GFA_TAG_STR_COMMENT_REM, &tag.expr))) {
        tag.tag = GFA_TAG_COMMENT;
      }  
    }
    // Recherche de signes
    else if (tag.tag==GFA_TAG_UNKNOWN) {
      buffer = GFA_Scanner_NormalizeSign(&tag, buffer);
      
      // Recherche de commentaire type ! ou '
      if ((tag.tag==GFA_TAG_SIGN) &&
          ((tag.subtag==GFA_SUBTAG_EXCLAMATION) ||
           ((tag.subtag==GFA_SUBTAG_APOST) && (!GFA_Tag_GetStackSize(stack_tag) || 
                                                ((*(GFA_String_GetStr(&tag.expr) - 1)==' ') && (!GFA_Charset_EofLine(*buffer)))))))  
      {
        tag.symbol = GFA_SYMBOL_COMMENT;
      }
    }
    
    // Commentaire
    if (tag.tag==GFA_TAG_COMMENT) {
      const char *begin = buffer;
      for (; !GFA_Charset_EofLine(*buffer); ++buffer);
      tag.expr = GFA_String_Create(begin, buffer - begin);  
    }
    // Caractre inconnu  
    else if (tag.tag==GFA_TAG_UNKNOWN)
      return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
    
    // Ajoute le tag dans la pile     
    if (!GFA_Tag_PushFromStack(stack_tag, &tag, -1))
      return GFA_GET_ERR(GFA_ERR_STACK_TAG_OVERFLOW, NULL, GFA_String_GetStr(&tag.expr));
  }
  
  // Ligne trop longue
  if ((buffer - line_begin) > GFA_BASIC_MAX_LINE_SIZE)
    return GFA_GET_ERR_N(GFA_ERR_LINE_TOO_LONG, NULL);
    
  // Ajoute tag marqueur fin de ligne
  tag.expr = GFA_String_Create(NULL, 0);
  tag.symbol = GFA_SYMBOL_EOFLINE;
  tag.struct_idx = -1;
  if (!GFA_Tag_PushFromStack(stack_tag, &tag, -1))
    return GFA_GET_ERR_N(GFA_ERR_STACK_TAG_OVERFLOW, NULL);
  
  // Stocke point d'arriv dans le buffer d'entr
  *rbuffer = buffer;
  
  // Finalise la pile
  if (!(err = GFA_Scanner_AdjustStack(stack_tag, stack_lib, stack_expr_find)).n) {
    if (!(err = GFA_Scanner_MakeNumber(stack_tag)).n) {
      if ((!(err = GFA_Scanner_DecryptParserCommands(stack_lib, stack_tag)).n) && GFA_Tag_GetStackSize(stack_tag)) {
        GFA_Scanner_FinalizeStack(stack_tag);
        err = GFA_Scanner_CheckSyntax(stack_tag);
      }
    }
  }
 
  return err;
}


GFA_Error GFA_Scanner_AdjustStack(GFA_StackTag *stack_tag, GFA_StackLib *stack_lib, GFA_StackExprFind *stack_expr_find) {
  GFA_Tag *tag;
  const GFA_Tag *preview_tag = NULL, *next_tag;
  short start_code = 0;
  bool use_mark_eofline = true;
  
  // Commande du parseur
  if (!GFA_Tag_GetParserAreaDefined(stack_tag)) {
    tag = GFA_Tag_GetFromStack(stack_tag, 0);
    if (tag->tag==GFA_TAG_SYMBOL) {
      // Recherche de commandes du parseur (INCLUDE...)
      if ((tag->info.index = GFA_Tbl_Parser_Find(&tag->expr))!=-1) {
    	  tag->symbol = GFA_SYMBOL_PARSER;
    	  
    	  // Les : ne sont plus considr comme marqueur fin de code
    	  use_mark_eofline = false;
      }
    }
    
    // Si on dcouvre une instruction, fin de la zone de commandes du parseur
    if ((tag->symbol!=GFA_SYMBOL_PARSER) && (tag->symbol!=GFA_SYMBOL_COMMENT) && (tag->symbol!=GFA_SYMBOL_EOFLINE))
      GFA_Tag_SetParserAreaDefined(stack_tag); 
  }
 
  // Parcours la pile
  for (short i = 0; i < (short)(GFA_Tag_GetStackSize(stack_tag) - 1); ++i) { // -1 -> On saute le tag EOFLINE
    if (i) preview_tag = GFA_Tag_GetFromStack(stack_tag, i - 1);
    tag = GFA_Tag_GetFromStack(stack_tag, i);
    next_tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
    
    /***********************************
     GESTION DES SYMBOLES
     **********************************/
    if ((tag->tag==GFA_TAG_SYMBOL) || (tag->tag==GFA_TAG_VAR)) {
      // Recherche de fonctions basiques (oprateurs logiques...)
      bool bSymFind = false;
      for (short i = (sizeof (GFA_Symbol_Table) / sizeof(struct s_Symbol_Table)) - 1; i>=0; --i) {
        if (!GFA_String_StrcmpUpper(GFA_Symbol_Table[i].expr, &tag->expr)) {
          tag->tag = GFA_Symbol_Table[i].tag;
          tag->subtag = GFA_Symbol_Table[i].subtag;
          bSymFind = true;
          break;
        }
      }
      if (bSymFind) continue;
      
      // Recherche de structures (IF, FOR ...)
      if ((tag->struct_idx = GFA_Structs_GetIndex(&tag->expr))!=-1) {
        tag->symbol = GFA_SYMBOL_STRUCT;
        continue;
      }
      
      // Recherche de fonctions
      if ((tag->info.library.index = GFA_Lib_FindFunctionInStack(&tag->info.library.object, stack_lib, &tag->expr))!=-1) {
      	tag->symbol = GFA_SYMBOL_FUNCTION;
        continue;
      }
            
      // Recherche de constantes
      if ((tag->info.library.index = GFA_Lib_FindConstantInStack(&tag->info.library.object, stack_lib, &tag->expr))!=-1) {
        tag->symbol = GFA_SYMBOL_CONSTANT;
        continue;
      }
    }
    
    // Retire suffixe d'une variable
    if (tag->tag==GFA_TAG_VAR)
      tag->expr = GFA_String_Create(GFA_String_GetStr(&tag->expr), GFA_String_GetSize(&tag->expr) - 1);
    // Fonction ou constante non trouve
    else if (tag->tag==GFA_TAG_SYMBOL) {
      // Fonction suffixe ? non trouve
      if (tag->subtag!=GFA_SUBTAG_NONE)
        return GFA_GET_ERR(GFA_ERR_SYNTAX_UNKNOWN_FUNCTION_OR_CONSTANT_NAME, NULL, GFA_String_GetStr(&tag->expr));
       
      // Goto | Restore label
      if ((preview_tag->struct_idx==GFA_STRUCT_GOTO) ||
          (preview_tag->struct_idx==GFA_STRUCT_RESTORE)) 
      {
        tag->symbol = GFA_SYMBOL_LABEL;
        tag->info.index = GFA_ExprFind_StringInStack(stack_expr_find, GFA_EXPR_LABEL, &tag->expr);
        continue;  
      } 
      // Label: simple
      if ((next_tag->symbol==GFA_SYMBOL_TWO_POINT) && 
          ((!i || (preview_tag->tag==GFA_TAG_EOFLINE)))) 
      {
        tag->symbol = GFA_SYMBOL_LABEL;
        tag->info.index = GFA_ExprFind_StringInStack(stack_expr_find, GFA_EXPR_LABEL, &tag->expr);
        GFA_Tag_DeleteFromStack(stack_tag, i + 1);    // Retire :
        continue;
      }
      
      // Forcment un type explicite
      if (GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VARIABLE_TYPE==GFA_SUBTAG_NOTHING)
        return GFA_GET_ERR(GFA_ERR_EXPLICIT_TYPE_FOR_VARIABLE, NULL, GFA_String_GetStr(&tag->expr));
      
      // Forcment une variable non suffixe (type par dfaut)
      tag->tag = GFA_TAG_VAR;
      tag->subtag = GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VARIABLE_TYPE;
    }
    
    // Recherche de variables 
    if (tag->tag==GFA_TAG_VAR) {
      // C'est un tableau ?
      if (next_tag->symbol==GFA_SYMBOL_BRACKET_OPEN)
        tag->tag = GFA_TAG_VARTABLE;
      
      tag->info.index = GFA_ExprFind_StringInStack(stack_expr_find, (tag->subtag - 1) + ((tag->tag==GFA_TAG_VARTABLE)?GFA_EXPR_TAB_U8:0), &tag->expr);
    }
    
    
    /***********************************
     GESTION DES SIGNES
     **********************************/
    // Conversion de : en EOFLINE
    else if ((tag->symbol==GFA_SYMBOL_TWO_POINT) && use_mark_eofline) {
      tag->symbol = GFA_SYMBOL_EOFLINE;
      start_code = (i + 1);
    }
    
    // Conversion du signe - en ngation
    else if (tag->symbol==GFA_SYMBOL_MATH_SUB) {
      if (// En dbut de ligne de code
          (i==start_code) ||
          // Fonction void type (myfunc -...)
          ((i==(start_code + 1)) && (preview_tag->symbol==GFA_SYMBOL_FUNCTION)) ||
          // Structure
          (preview_tag->tag==GFA_TAG_STRUCT) ||
          // ,|;|'|(
          ((preview_tag->symbol==GFA_TAG_SIGN) && (preview_tag->subtag!=GFA_SUBTAG_BRACKET_CLOSE)) ||
          // Signes maths, logiques ou de comparaisons
          ((preview_tag->tag==GFA_TAG_MATH) || (preview_tag->tag==GFA_TAG_LOGIC) || (preview_tag->tag==GFA_TAG_COMPARE)))
       {  
         tag->symbol = GFA_SYMBOL_MATH_NEGATIF;
       }  
    }
  }
  
  return GFA_GET_ERR_OK;
}


GFA_Error GFA_Scanner_MakeNumber(GFA_StackTag *stack_tag) {
  GFA_Tag *tag;
  const GFA_Tag *next_tag;
  bool negatif;
    
  for (short i = (short)(GFA_Tag_GetStackSize(stack_tag) - 2); i >= 0; --i) {
    tag = GFA_Tag_GetFromStack(stack_tag, i);
    next_tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
    
    // Chane de caractres, aucune manipulation
    if (tag->symbol==GFA_SYMBOL_VALUE_STRING)
      continue;
    
    // Conversion en flottant d'une expression
    if (tag->tag==GFA_TAG_BASE) {
      tag->info.number.value_float = (float)tag->info.number.value_S32;
    }
    else if (tag->tag==GFA_TAG_VALUE) {
      // Conversion de tous les types en flottant
      if (is_transfinite(tag->info.number.value_float = GFA_String_StrToFloat(&tag->expr)))
        return GFA_GET_ERR(GFA_ERR_NUMBER_OVERFLOW, NULL, GFA_String_GetStr(&tag->expr));
        
      // Conversion spcialise
      if (tag->subtag!=GFA_SUBTAG_FLOAT) {
        // Nombre avec flottant, conversion impossible
        if (floor(tag->info.number.value_float)!=tag->info.number.value_float)
          return GFA_GET_ERR(GFA_ERR_INVALID_NUMBER_CONVERSION, NULL, GFA_String_GetStr(&tag->expr));
        
        // Conversion en long et vrification overflow
        tag->info.number.value_S32 = (long)tag->info.number.value_float;
        if ((float)tag->info.number.value_S32!=tag->info.number.value_float)
          return GFA_GET_ERR(GFA_ERR_NUMBER_OVERFLOW, NULL, GFA_String_GetStr(&tag->expr));
      }
    }
    
    // Vrification overflow pour les autres plages de donnes
    if ((tag->tag==GFA_TAG_VALUE) || (tag->tag==GFA_TAG_BASE)) {
      
      switch (tag->subtag) {
        case GFA_SUBTAG_INT_U8: 
        {
          if ((tag->info.number.value_S32 < 0) || (tag->info.number.value_S32 > 0xFF))
            return GFA_GET_ERR(GFA_ERR_NUMBER_OVERFLOW, NULL, GFA_String_GetStr(&tag->expr));
          break;
        }
          
        case GFA_SUBTAG_INT_S16:
        {
          if ((tag->info.number.value_S32 < SHRT_MIN) || (tag->info.number.value_S32 > SHRT_MAX))
            return GFA_GET_ERR(GFA_ERR_NUMBER_OVERFLOW, NULL, GFA_String_GetStr(&tag->expr));
          break;
        }
        
        default:;
      } 
      
      continue; 
    }
   
    // Expression ngative
    if ((tag->symbol==GFA_SYMBOL_MATH_NEGATIF) &&
        (((next_tag->tag==GFA_TAG_VALUE) || (next_tag->tag==GFA_TAG_BASE)) && 
        ((next_tag->subtag!=GFA_SUBTAG_STRING) && (next_tag->subtag!=GFA_SUBTAG_INT_U8))))
    {
      negatif = false;
      do {
        GFA_Tag_DeleteFromStack(stack_tag, i--);
        negatif = !negatif;
        if (i>=0) tag = GFA_Tag_GetFromStack(stack_tag, i);
        else break;
      } while (tag->symbol==GFA_SYMBOL_MATH_NEGATIF);
      
      if (negatif) {
        tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
        tag->info.number.value_float = -tag->info.number.value_float;
        if (tag->subtag!=GFA_SUBTAG_FLOAT)
          tag->info.number.value_S32 = -tag->info.number.value_S32;
      }
      
      ++i;
      continue;  
    }
  }
  
  return GFA_GET_ERR_OK;
}


// Gre les commandes du parseur
GFA_Error GFA_Scanner_DecryptParserCommands(GFA_StackLib *stack_lib, GFA_StackTag *stack_tag) {
	GFA_ERROR_NUMBER err_n;
	const GFA_Tag *tag = GFA_Tag_GetFromStack(stack_tag, 0);
	unsigned short stack_size = GFA_Tag_GetStackSize(stack_tag) - 1;
	
	if (tag->symbol==GFA_SYMBOL_PARSER) {
		const GFA_Tag *next_tag = GFA_Tag_GetFromStack(stack_tag, 1);
		switch (tag->info.index) {
			// *************************
			// Ouverture d'une librairie
			// *************************
			case GFA_PARSER_INCLUDE: 
			{
				// Vrification de la syntaxe
				if ((stack_size!=2) || (next_tag->symbol!=GFA_SYMBOL_VALUE_STRING))
				  return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
				
				// Ouverture de la librairie
				GFA_Lib include_lib;
				if ((err_n = GFA_Lib_Open(&include_lib, GFA_String_GetCString(&next_tag->expr))))
				  return GFA_GET_ERR_N(err_n, NULL);
				
				// Ajoute la librairie dans la pile
				if (!GFA_Lib_PushLib(stack_lib, &include_lib))
				  return GFA_GET_ERR_N(GFA_ERR_STACK_LIB_OVERFLOW, NULL);
				
				break;
			}
			
			
			// ********************************
			// Type des variables non suffixes
			// ********************************
			case GFA_PARSER_VARIABLE_NOT_SUFFIXED:
			{
				if (stack_size==3) {
					const GFA_Tag *end_tag = GFA_Tag_GetFromStack(stack_tag, 2);
					if ((next_tag->struct_idx==GFA_STRUCT_AS) && 
					    (end_tag->struct_idx>=GFA_STRUCT_BYTE) && (end_tag->struct_idx<=GFA_STRUCT_NOTHING))
					{
						GFA_BASIC_PARAMETERS.DEFAULT_NOT_SUFFIXED_VARIABLE_TYPE = (end_tag->struct_idx - GFA_STRUCT_BYTE) + GFA_SUBTAG_INT_U8; 
						break;
					}
					
				}
				
				return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
			}
			
			default:
			  Assert(false);
		}	
	}
	
	return GFA_GET_ERR_OK;
} 


void GFA_Scanner_FinalizeStack(GFA_StackTag *stack_tag) {
  // Nous sommes dans la zone des commandes du parseur, ne pas modifier la pile
  if (!GFA_Tag_GetParserAreaDefined(stack_tag))
    return;
  
  GFA_Tag *tag;
  const GFA_Tag *next_tag;
  
  bool egal_affec = true;
  
  for (short i = 0; i < (short)(GFA_Tag_GetStackSize(stack_tag) - 1); ++i) {
    tag = GFA_Tag_GetFromStack(stack_tag, i);
    next_tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
       
    // Signe -> d'affectation
    if (tag->symbol==GFA_SYMBOL_MATH_AFFEC_RIGHT) {
    	egal_affec = false;
    	continue;
    }
    
    // Remplace = par signe d'affectation
    if ((tag->symbol==GFA_SYMBOL_COMPARE_EGAL) && egal_affec) {
      tag->symbol = GFA_SYMBOL_MATH_AFFEC_LEFT;
      egal_affec = false;
      continue;
    }
    
    // Remplace Else + If par ElseIf
    if ((tag->struct_idx==GFA_STRUCT_ELSE) && (next_tag->struct_idx==GFA_STRUCT_IF)) {
      tag->struct_idx = GFA_STRUCT_ELSEIF;
      GFA_Tag_DeleteFromStack(stack_tag, i + 1);
      continue;
    }
    // Remplace Exit + If par "ExitIf"
    if ((tag->struct_idx==GFA_STRUCT_EXIT) && (next_tag->struct_idx==GFA_STRUCT_IF)) {
      tag->struct_idx = GFA_STRUCT_EXITIF;
      GFA_Tag_DeleteFromStack(stack_tag, i + 1);
      continue;
    }
    
    // EOFLINE
    if (!i || (tag->symbol==GFA_SYMBOL_EOFLINE) || (tag->struct_idx==GFA_STRUCT_THEN)) {
      if (i) tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
      egal_affec = (tag->tag==GFA_TAG_VAR) || (tag->tag==GFA_TAG_VARTABLE) || 
                   (tag->struct_idx==GFA_STRUCT_FOR);
      continue;
    }  
  }
}


GFA_Error GFA_Scanner_CheckSyntax(GFA_StackTag *stack_tag) {
  // Nous sommes dans la zone des commandes du parseur, ne pas vrifier la syntaxe
  if (!GFA_Tag_GetParserAreaDefined(stack_tag))
    return GFA_GET_ERR_OK;
  
  const GFA_Tag *tag, *end_tag;
  const GFA_Tag *next_tag = NULL, *preview_tag = NULL;
  
  unsigned short start_code = 0;
  unsigned short size_code;
  short first_struct_idx = -1;
  
  bool find_then = false;
  unsigned short find_to = 0;
  bool find_down_to = false;
  unsigned short find_step = 0;
  
  unsigned short size_stack = GFA_Tag_GetStackSize(stack_tag);
  for (unsigned short i = 0; i < size_stack; ++i) {
    if (i) preview_tag = GFA_Tag_GetFromStack(stack_tag, i - 1);
    tag = GFA_Tag_GetFromStack(stack_tag, i);
    if (i < (size_stack - 1)) 
      next_tag = GFA_Tag_GetFromStack(stack_tag, i + 1);
    
    // ***********************************************************
    // Fin d'une ligne de code
    // ***********************************************************
    if (tag->tag==GFA_TAG_EOFLINE) {
      size_code = i - start_code;
      if (size_code) {
        end_tag = GFA_Tag_GetFromStack(stack_tag, start_code + 1);
        
        // Vrification structure en dbut de ligne
        switch (first_struct_idx) {
          // Structures dj vrifies
          case -1: 
          case GFA_STRUCT_PRINT:
          {
            break;
          }
          
          // Goto Label
          case GFA_STRUCT_GOTO:
          {
            if ((end_tag->symbol==GFA_SYMBOL_LABEL) && (size_code==2))
              break;
              
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
          }
          
          // Restore | Restore Label
          case GFA_STRUCT_RESTORE:
          {
            if ((size_code==1) ||
                ((end_tag->symbol==GFA_SYMBOL_LABEL) && (size_code==2)))
              break;
              
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
          }
          
          // If ... Then [...]
          // ELSEIF ... Then
          case GFA_STRUCT_IF:
          case GFA_STRUCT_ELSEIF:
          {
            if (find_then)
              break;
            
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
          }
          
          // Structure seule sur la ligne
          // Else | Endif | Exit |Continue
          // Do | Loop
          // Wend
          // Repeat
          case GFA_STRUCT_ELSE:  case GFA_STRUCT_ENDIF:
          case GFA_STRUCT_EXIT:  case GFA_STRUCT_CONTINUE:
          case GFA_STRUCT_DO:    case GFA_STRUCT_LOOP:
          case GFA_STRUCT_WEND:  case GFA_STRUCT_REPEAT:
          {
            if (size_code==1)
              break;
              
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);  
          }
          
          // Structure forcment suivie d'une expression  
          // ExitIf
          // While | Until
          // Data | Read
          case GFA_STRUCT_EXITIF:
          case GFA_STRUCT_WHILE:
          case GFA_STRUCT_UNTIL:
          case GFA_STRUCT_DATA:
          case GFA_STRUCT_READ:
          {
            if (size_code>1)
              break;
            
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);  
          }
            
          // For ... To ... [Step] ...
          // For ... DownTo
          case GFA_STRUCT_FOR:
          {
            if ((find_step && ((i - find_step) < 2)) || !find_to)
              return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL); 
            
            /*// Dbut de la forme var=... 
            tag = GFA_Tag_GetFromStack(stack_tag, start_code + 1);
            next_tag = GFA_Tag_GetFromStack(stack_tag, start_code + 2);
            if (((tag->tag==GFA_TAG_VAR) && (tag->subtag<=GFA_SUBTAG_INT_S32)) &&
                (next_tag->symbol==GFA_SYMBOL_MATH_AFFEC_LEFT))
              break;
            
            // Fin de la forme ...->var
            tag = GFA_Tag_GetFromStack(stack_tag, find_to - 1);
            preview_tag = GFA_Tag_GetFromStack(stack_tag, find_to - 2);
            if (((tag->tag==GFA_TAG_VAR) && (tag->subtag<=GFA_SUBTAG_INT_S32)) &&
                (preview_tag->symbol==GFA_SYMBOL_MATH_AFFEC_RIGHT_NO_CLEAR))
              break;
              
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);*/
            
            break; 
          }
          
          // Next  
          case GFA_STRUCT_NEXT:
          {
            // Next seul ou suivi d'une variable
            if ((size_code==1) ||
                ((size_code==2) && (end_tag->tag==GFA_TAG_VAR) && (end_tag->subtag>=GFA_SUBTAG_INT_U8) && (end_tag->subtag<=GFA_SUBTAG_FLOAT)))
              break;  
            
            return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
          }
          
          default:
            return GFA_GET_ERR_N(GFA_ERR_INTERNAL_UNKNOWN_STRUCTURE, NULL);
        }
      }
      
      start_code = i + 1;
      find_then = false;
      find_to = find_step = 0;
      find_down_to = false;
      first_struct_idx = -1;
      continue;
    }
    
    
    // ***********************************************************
    // Structure n'importe o
    // ***********************************************************
    switch (tag->struct_idx) {
      case -1: break;
      
      // Structures en dbut de ligne de code seulement
      case GFA_STRUCT_IF:     case GFA_STRUCT_ELSE:  case GFA_STRUCT_ELSEIF: case GFA_STRUCT_ENDIF:
      case GFA_STRUCT_DO:     case GFA_STRUCT_LOOP:
      case GFA_STRUCT_WHILE:  case GFA_STRUCT_WEND:
      case GFA_STRUCT_REPEAT: case GFA_STRUCT_UNTIL:
      case GFA_STRUCT_FOR:    case GFA_STRUCT_NEXT:
      case GFA_STRUCT_DATA:   case GFA_STRUCT_READ:
      {
        if ((i!=start_code) || find_then)
          return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
        
        first_struct_idx = tag->struct_idx;
        continue;
      }
      
      // Structures en dbut de ligne de code ou aprs un If ... Then
      case GFA_STRUCT_GOTO:
      case GFA_STRUCT_EXIT:   case GFA_STRUCT_EXITIF: case GFA_STRUCT_CONTINUE:
      case GFA_STRUCT_RESTORE:
      case GFA_STRUCT_PRINT:
      {
      	if (i!=start_code)
          return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
        
        first_struct_idx = tag->struct_idx;
      	continue;
      }
      
      // At
      case GFA_STRUCT_AT:
      {
      	if ((first_struct_idx==GFA_STRUCT_PRINT) && (i==(start_code + 1)) && (next_tag->symbol==GFA_SYMBOL_BRACKET_OPEN))
      	  continue;
      	return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
      }
            
      // If ... Then [...]
      // ElseIf ... Then
      case GFA_STRUCT_THEN:
      {
        if (((first_struct_idx==GFA_STRUCT_IF) || 
            ((first_struct_idx==GFA_STRUCT_ELSEIF) && (next_tag->tag==GFA_TAG_EOFLINE))) &&
            !find_then &&
            (i - start_code)>=2)
        {
          find_then = true;
          
          start_code = i + 1;
          first_struct_idx = -1;
          
          break;
        }
        
        return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
      }
      
      // DownTo | To
      case GFA_STRUCT_DOWNTO:
        find_down_to = true;
      case GFA_STRUCT_TO:
      {
        if ((first_struct_idx==GFA_STRUCT_FOR) && ((i - start_code)>=2) && !find_to && !find_step) {
          find_to = i;
          break;
        }
        
        return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
      }
      
      // Step
      case GFA_STRUCT_STEP:
      {
        if ((first_struct_idx==GFA_STRUCT_FOR) && !find_step && find_to && !find_down_to && ((i - find_to)>=2)) {
          find_step = i;
          break;
        }
        
        return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
      }

      // Structures non gres
      default:
        //return GFA_GET_ERR_N(GFA_ERR_INTERNAL_UNKNOWN_STRUCTURE, NULL);
        return GFA_GET_ERR_N(GFA_ERR_SYNTAX_GENERAL, NULL);
    }
  }
  
  return GFA_GET_ERR_OK;
}
