// C Source File
// Created 18/08/2006; 16:24:39

#include "tigcclib.h"
#include "tbl_tags.h"
#include "tbl_parser.h"
#include "tbl_structs.h"
#include "memory.h"
#include "charset.h"
#include "gfalib.h"
#include "tags.h"

GFA_StackTag GFA_Tag_CreateStack() {
  GFA_StackTag stack;
  stack.item = GFA_Memory_Alloc(GFA_TAG_STACK_SIZE * sizeof(GFA_Tag));
  stack.size = 0;
  stack.parser_area_defined = false;
  return stack;
}


inline void GFA_Tag_DeleteStack(GFA_StackTag *stack) {
  Assert(stack);
  
  GFA_Memory_Free(stack->item);
}


inline bool GFA_Tag_GetParserAreaDefined(const GFA_StackTag *stack) {
	Assert(stack);
	
	return stack->parser_area_defined;
}


inline void GFA_Tag_SetParserAreaDefined(GFA_StackTag *stack) {
	Assert(stack);
	
	stack->parser_area_defined = true;
}


GFA_Tag GFA_Tag_Create(const GFA_String *expr, GFA_TAG tag, GFA_SUBTAG subtag, GFA_PRIORITY priority) {
  GFA_Tag res;
  res.expr = expr?*expr:GFA_String_Create(NULL, 0);
  res.tag = tag;
  res.subtag = subtag;
  res.treetag = GFA_TREETAG_NONE;
  res.priority = priority;
  res.struct_idx = -1;
  return res;
}


bool GFA_Tag_PushFromStack(GFA_StackTag *stack, const GFA_Tag *tag, short pos) {
  Assert(stack && tag);
  
  #if _DEBUG
    if (pos!=-1)
      Assert((pos>=0) && (pos < (short)stack->size));
  #endif 
    
  if (stack->size>=GFA_TAG_STACK_SIZE)
    return false;
  
  if (pos==-1)
    pos = stack->size;
  else
    memmove(&stack->item[pos + 1], &stack->item[pos], (&stack->item[stack->size] - &stack->item[pos]) * sizeof(GFA_Tag));
  
  stack->item[pos] = *tag;
  ++stack->size;
    
  return true;
}


void GFA_Tag_DeleteFromStack(GFA_StackTag *stack, unsigned short n) {
  Assert(stack && (n < (stack->size)));
  
  memmove(&stack->item[n], &stack->item[n + 1], (stack->size-- - n) * sizeof(GFA_Tag));
}


inline GFA_Tag *GFA_Tag_GetFromStack(const GFA_StackTag *stack, unsigned short n) {
  Assert(stack && (n < (stack->size)));
  
  return &stack->item[n];
}


inline unsigned short GFA_Tag_GetStackSize(const GFA_StackTag *stack) {
  Assert(stack);
  
  return stack->size;
}


inline void GFA_Tag_ClearStack(GFA_StackTag *stack) {
	Assert(stack);
	
	stack->size = 0;
}


bool GFA_Tag_PopTagFromStack(GFA_StackTag *stack, GFA_Tag **tag, bool decrease) {
  Assert(stack && tag);
  
  if (!stack->size)
    return false;
  
  *tag = &stack->item[stack->size - 1];  
  if (decrease)
    --stack->size;
  
  return true;
}


inline void GFA_Tag_DeleteNItemsFromStack(GFA_StackTag *stack, unsigned short n) {
	Assert(stack && (n<=GFA_Tag_GetStackSize(stack)));

	stack->size -= n;
}


#ifdef _DEBUG
  void GFA_Tag_PrintStack(const GFA_StackLib *stack_lib, const GFA_StackTag *stack_tag) {
    Assert(stack_lib && stack_tag);
    
    static const char *TableSymbols_math[] = {
      "=", "->", "+", "-", "*", "/", "\\", "^", " MOD ", "\xAD"
    };
    
    static const char *TableSymbols_logic[] = {
    	"NOT ", " AND ", " OR ", " XOR ", " IMP ", " EQV "
    };
    
    static const char *TableSymbols_compare[] = {
    	"==", "<", ">", "<=", ">=", "<>"
    };
    
    static const char *TableSymbols_sign[] = {
    	"(", ")", ".", ";", "'", ":", "!"
    };
    
    static const char TableSymbol_Suffix[] = {'|', '&', '%', '#', '!', '$'};
        
    const GFA_Tag *tag;
    const char *sstr = NULL;
    
    for (unsigned short i = 0; i < GFA_Tag_GetStackSize(stack_tag); ++i) {
      tag = GFA_Tag_GetFromStack(stack_tag, i);
      
      switch (tag->tag) {      	
      	case GFA_TAG_PARSER:
      	{
      		if (i) printf(" ");
      		printf("%s ", GFA_TBL_PARSER[tag->info.index]);
      		continue;
      	}
      	
      	case GFA_TAG_COMMENT:
      	{
      		printf("REM %s", GFA_String_GetCString(&tag->expr));
      		continue;
      	}
      	
      	case GFA_TAG_VALUE:
      	case GFA_TAG_BASE:
      	{
      	  if (tag->subtag==GFA_SUBTAG_STRING)
      	    printf("\"%s\"", GFA_String_GetCString(&tag->expr));
      	  else
      	    printf("%f", tag->info.number.value_float);
      	  
      		continue;
      	}
      	
      	case GFA_TAG_MATH:
      	{
      		sstr = TableSymbols_math[tag->subtag - 1];
      		if (tag->treetag==GFA_TREETAG_DONT_CLEAR_STACK)
      		  printf("\\");
      		break;
      	}
      	
      	case GFA_TAG_LOGIC:
      	{
      		sstr = TableSymbols_logic[tag->subtag - 1];
      		break;
      	}
      	
      	case GFA_TAG_COMPARE:
      	{
      		sstr = TableSymbols_compare[tag->subtag - 1];
      		break;
      	}
      	
      	case GFA_TAG_SIGN:
      	{
      	  sstr = TableSymbols_sign[tag->subtag - 1];
         	break;
      	}
      	
      	case GFA_TAG_STRUCT:
      	{
      		if (i) printf(" ");
      		printf("%s ", GFA_Structs_GetName(tag->struct_idx));
      		continue;
      	}
      	
      	case GFA_TAG_VAR:
      	{
      		printf("var%d%c", tag->info.index, TableSymbol_Suffix[tag->subtag - 1]);
      		continue;
      	}
      	
      	case GFA_TAG_VARTABLE:
      	{
      		printf("tab%d%c", tag->info.index, TableSymbol_Suffix[tag->subtag - 1]);
      		continue;
      	}
      	
      	case GFA_TAG_LABEL:
      	{
      		printf("label%d", tag->info.index);
      		continue;
      	}
      	
      	case GFA_TAG_SYMBOL:
      	{
      		const GFA_Lib *lib = GFA_Lib_GetLib(stack_lib, tag->info.library.index);
      	  GFA_String str = (tag->subtag==GFA_SUBTAG_CONSTANT)?
      	                   GFA_Lib_GetConstantName(lib, &tag->info.library.object):
      	                   GFA_Lib_GetFunctionName(lib, &tag->info.library.object);
      		printf("%s ", GFA_String_GetCString(&str));  
      		continue;
      	}
      	
      	case GFA_TAG_EOFLINE:
      	{
      	  printf(":");
      	  continue;
      	}
      	
      	case GFA_TAG_ENDTAG:
      	{
      		printf("~");
      	  continue;
      	}
      	
      	default:
      	  printf("[unknown]");
      }
      
      printf("%s", sstr);
    }  
    
    printf("\n");
  }
#endif
