// C Source File
// Created 26/07/2006; 18:07:52

#include "tigcclib.h"
#include "tbl_tags.h"
#include "tbl_tokens.h"
#include "gfabasic.h"
#include "tokenisor.h"
#include "errors.h"
#include "files.h"
#include "charset.h"
#include "expr_find.h"
#include "scanner.h"
#include "utilities.h"

const char GFA_TOKENISOR_HEADER_SIGNATURE[6] = "GFABas";
const unsigned char GFA_TOKENISOR_HEADER_VERSION[2] = {2, 0};

GFA_TokensBuffer GFA_Tokenisor_CreateBuffer() {
  GFA_TokensBuffer result = GFA_MemoryBuffer_Create();
  
  // Ecriture en-tte fichier
  GFA_MemoryBuffer_WriteData(&result, GFA_TOKENISOR_HEADER_SIGNATURE, sizeof(GFA_TOKENISOR_HEADER_SIGNATURE));
  GFA_MemoryBuffer_WriteData(&result, GFA_TOKENISOR_HEADER_VERSION,   sizeof(GFA_TOKENISOR_HEADER_VERSION));
  GFA_MemoryBuffer_WriteShort(&result, 0x0000); // Jump_offset
    
  return result;
}


inline void GFA_Tokenisor_DeleteBuffer(GFA_TokensBuffer *tks_buff) {
  GFA_MemoryBuffer_Delete(tks_buff);
}


void GFA_Tokenisor_WriteSymbols(GFA_TokensBuffer *tks_buff, const GFA_StackExprFind *stack_expr_find) {
  for (unsigned short i = 0; i < GFA_EXPR_NUMBER_OF_TYPE; ++i) {
    const GFA_String *stack_strs = GFA_ExprFind_GetStringsFromType(stack_expr_find, i);
    for (unsigned short j = 0; j < GFA_ExprFind_GetStackSize(stack_expr_find, i); ++j) {
      GFA_MemoryBuffer_WriteByte(tks_buff, GFA_String_GetSize(&stack_strs[j]));
      GFA_MemoryBuffer_WriteData(tks_buff, GFA_String_GetStr(&stack_strs[j]), GFA_String_GetSize(&stack_strs[j]));
    }
  } 
}



void GFA_Tokenisor_WriteStackTag(GFA_TokensBuffer *tks_buff, const GFA_StackTag *stack_tag) {
  short stack_size = GFA_Tag_GetStackSize(stack_tag) - 1;
  const GFA_Tag *tag;
  unsigned short data_size;
  
  for (short i = 0; i < stack_size; ++i) {
    tag = GFA_Tag_GetFromStack(stack_tag, i);
    
    // Cration du token d'identification
    GFA_MemoryBuffer_WriteByte(tks_buff, (tag->tag==GFA_TAG_EOFLINE)?GFA_TOKEN_EOFCODE:MakeByte(tag->tag - 1, tag->subtag));
    
    
    // *************************************
    // Ecriture des donnes (complte Token)
    // *************************************
    switch (tag->tag) {
      // Commentaire
      // Chane de caractres
      case GFA_TAG_COMMENT:
      {
        GFA_MemoryBuffer_WriteShort(tks_buff, data_size = GFA_String_GetSize(&tag->expr));      // Taille de la chane
        GFA_MemoryBuffer_WriteData(tks_buff, (void *)GFA_String_GetStr(&tag->expr), data_size); // Donnes
        continue;
      }
           
      // Tableaux
      // Variables
      // Label
      // Commande du parseur
      case GFA_TAG_VARTABLE:
      //  ++i;  // Force  sauter la parenthse ouvrante
      case GFA_TAG_VAR:
      case GFA_TAG_LABEL:
      case GFA_TAG_PARSER:
      {
        GFA_MemoryBuffer_WriteByte(tks_buff, tag->info.index);                                  // Index
        continue;
      }
            
      // Structure
      case GFA_TAG_STRUCT:
      {
        GFA_MemoryBuffer_WriteByte(tks_buff, tag->struct_idx);                                  // Index de la structure
        continue;
      }
      
      // Fonction ou constante
      case GFA_TAG_SYMBOL:
      {
        GFA_MemoryBuffer_WriteByte(tks_buff, tag->info.library.index);                                 // Numro de librairie
        GFA_MemoryBuffer_WriteByte(tks_buff, GFA_Lib_Object_GetTokenIndex(&tag->info.library.object)); // Index de la donne (on en dduit le type pour une constante)
        continue;
      }
      
      // Nombres
      case GFA_TAG_VALUE:
      case GFA_TAG_BASE:
      {
        switch (tag->subtag) {          
          case GFA_SUBTAG_INT_U8:
          {
            GFA_MemoryBuffer_WriteByte(tks_buff, (unsigned char)tag->info.number.value_S32);
            continue;
          }
          
          case GFA_SUBTAG_INT_S16:
          {
            GFA_MemoryBuffer_WriteShort(tks_buff, (short)tag->info.number.value_S32);
            continue;
          }
          
          case GFA_SUBTAG_INT_S32:
          {
            GFA_MemoryBuffer_WriteLong(tks_buff, tag->info.number.value_S32);
            continue;
          }
          
          case GFA_SUBTAG_FLOAT: 
          {
            GFA_MemoryBuffer_WriteData(tks_buff, (void *)&tag->info.number.value_float, sizeof(float));
            continue;
          }
          
          case GFA_SUBTAG_STRING:
          {
          	GFA_MemoryBuffer_WriteShort(tks_buff, data_size = GFA_String_GetSize(&tag->expr));      // Taille de la chane
            GFA_MemoryBuffer_WriteData(tks_buff, (void *)GFA_String_GetStr(&tag->expr), data_size); // Donnes
            continue;
          }
          
          // Erreur interne
          default:
            Assert(false); 
        }
      }
      
      // Tag sans donnes (signes (mathmatiques))...
      default:;
    }
  }
  
  // Fin de ligne
  GFA_MemoryBuffer_WriteByte(tks_buff, GFA_TOKEN_EOFLINE);
}


GFA_ERROR_NUMBER GFA_Tokenisor_CreateFile(GFA_TokensBuffer *tks_buff, const char *file, const GFA_StackExprFind *stack_expr_find) {
  // Token marqueur fin de fichier
  GFA_MemoryBuffer_WriteByte(tks_buff, GFA_TOKEN_EOFFILE);
  
  void *fpos = GFA_MemoryBuffer_GetBeginAddress(tks_buff);
  
  // Ecriture jump offset
  *(unsigned short *)(fpos + sizeof(GFA_Tokenisor_Header) - sizeof(unsigned short)) = GFA_MemoryBuffer_GetCurrentAddress(tks_buff) - fpos;
  
  // Informations table des symboles
  for (unsigned short i = 0; i < GFA_EXPR_NUMBER_OF_TYPE; ++i)
    GFA_MemoryBuffer_WriteByte(tks_buff, GFA_ExprFind_GetStackSize(stack_expr_find, i));
  
  // Ecriture de la table des symbols
  GFA_Tokenisor_WriteSymbols(tks_buff, stack_expr_find);
  
  return GFA_File_GenerateFileFromMemoryBuffer(tks_buff, file, GFA_TOKENIZED_FILE_EXTENSION);
}


bool GFA_Tokenisor_EofFile(const GFA_File *file) {
	Assert(file);
	return (*file->ptr==(char)GFA_TOKEN_EOFFILE);
}


GFA_ERROR_NUMBER GFA_Tokenisor_ExtractHeader(GFA_File *file, GFA_StackExprFind *stack_expr_find) {
	Assert(file && stack_expr_find && !GFA_File_Eof(file));
	
	if (GFA_File_GetType(file)!=OTH_TAG)
	  return GFA_ERR_INCORRECT_FILE_TYPE;
	
	// Vrification en-tte du fichier
	GFA_Tokenisor_Header *header = (GFA_Tokenisor_Header *)file->data;
	if (memcmp(GFA_TOKENISOR_HEADER_SIGNATURE, header->signature, sizeof(GFA_TOKENISOR_HEADER_SIGNATURE)))
	  return GFA_ERR_INCORRECT_FILE_SIGNATURE;
	if (memcmp(GFA_TOKENISOR_HEADER_VERSION, header->version, sizeof(GFA_TOKENISOR_HEADER_VERSION)))
	  return GFA_ERR_INCORRECT_FILE_VERSION;
	
	// Lecture de la table des symboles
	const char *sym_ptr = (const char *)(file->data + header->jump_offset);
	GFA_Tokenisor_TableSymbolsInfo tSym;
	memcpy(&tSym, sym_ptr, sizeof(GFA_Tokenisor_TableSymbolsInfo));
	sym_ptr += sizeof(GFA_Tokenisor_TableSymbolsInfo);
		
	// Remplie la pile contenant le nom de chaque symbol
	for (unsigned short i = 0; i < GFA_EXPR_NUMBER_OF_TYPE; ++i) {	
		while (((unsigned char *)&tSym)[i]--) {
		  GFA_String sstr = GFA_String_Create(sym_ptr + 1, *sym_ptr);
			if (!GFA_ExprFind_PushString(stack_expr_find, i, &sstr))
			  return GFA_ERR_STACK_EXPRFIND_OVERFLOW;
			  
			sym_ptr += (*sym_ptr + 1);
		}
	}
	
	// Saute en-tte
	file->ptr += sizeof(GFA_Tokenisor_Header);
	
	return GFA_ERR_OK;
}


GFA_ERROR_NUMBER GFA_Tokenisor_ExtractData(GFA_File *file, unsigned short *line, const GFA_StackLib *stack_lib, GFA_StackTag *stack_tag) {
	Assert(file && stack_tag);
	
	GFA_Tag tag;
	unsigned char etag;
	
	while (!GFA_File_Eof(file)) {
		switch ((etag = *(file->ptr++))) {
			// Marqueur fin code
			case GFA_TOKEN_EOFLINE:
			  ++*line;
			case GFA_TOKEN_EOFCODE:
			{
			  tag = GFA_Tag_Create(NULL, GFA_TAG_EOFLINE, GFA_SUBTAG_NONE, GFA_PRIORITY_NONE);
			  return GFA_Tag_PushFromStack(stack_tag, &tag, -1)?
			         GFA_ERR_OK:
			         GFA_ERR_STACK_TAG_OVERFLOW;
			}
			  
			// Dcodage du tag  		
			default:
			{
				tag = GFA_Tag_Create(NULL, (etag & 0x0F) + 1, (etag >> 4), GFA_PRIORITY_NONE);
			
				switch (tag.tag) {
					// Tags sans donnes
					case GFA_TAG_MATH:
					case GFA_TAG_LOGIC:
          case GFA_TAG_COMPARE:
          case GFA_TAG_SIGN:
            break;
					
					// Commentaire
          // Chane de caractres
					case GFA_TAG_COMMENT:
					{
						unsigned short size = (unsigned short)*(file->ptr++) << 8;
						size |= *(file->ptr++);
						
						tag.expr = GFA_String_Create(file->ptr, size);
						file->ptr += size;	
						continue;
					}
					
					// Commande du parseur
					case GFA_TAG_PARSER:
					{
						tag.info.index = *(unsigned char *)(file->ptr++);
						break;
					}
					
					// Tableaux
          // Variables
          // Label
          case GFA_TAG_VARTABLE:
          case GFA_TAG_VAR:
          case GFA_TAG_LABEL:
          {
          	tag.info.index = *(unsigned char *)(file->ptr++);
          	break;
          }
									
					// Structure
					case GFA_TAG_STRUCT:
					{
						tag.struct_idx = *(unsigned char *)(file->ptr++);
						break;
					}
					
					// Fonction ou constante
					case GFA_TAG_SYMBOL:
					{
					  const GFA_Lib *lib = GFA_Lib_GetLib(stack_lib, (tag.info.library.index = *(unsigned char *)(file->ptr++)));
					  unsigned char token_idx = *(unsigned char *)(file->ptr++);
					  bool is_good = (tag.subtag==GFA_SUBTAG_CONSTANT)?
					                 GFA_Lib_GetConstantObjectFromToken(&tag.info.library.object, lib, token_idx):
					                 GFA_Lib_GetFunctionObjectFromToken(&tag.info.library.object, lib, token_idx);
					  
					  // ? Message d'erreur
					  if (!is_good) {
					  	Assert(false);
					  }               
					                 
					  // Constante
					  if (tag.subtag==GFA_SUBTAG_CONSTANT) {
					  	const void *data = GFA_Lib_GetConstantData(lib, &tag.info.library.object);
						  
						  switch (GFA_Lib_Object_GetConstantType(&tag.info.library.object)) {						  	
						  	case GFA_CONSTANT_INTEGER:
						  	{
						  		tag.tag = GFA_TAG_VALUE;
						  		tag.subtag = GFA_SUBTAG_INT_S32;
						  		tag.info.number.value_S32 = *(long *)data;
						  		tag.info.number.value_float = (float)tag.info.number.value_S32;
						  		break;
						  	}
						  	
						  	case GFA_CONSTANT_FLOAT:
						  	{
						  		tag.tag = GFA_TAG_VALUE;
						  		tag.subtag = GFA_SUBTAG_FLOAT;
						  		tag.info.number.value_float = *(float *)data;
						  		tag.info.number.value_S32 = (long)tag.info.number.value_float;
						  		break;
						  	}
						  	
						  	case GFA_CONSTANT_STRING:
						  	{
						  		tag.symbol = GFA_SYMBOL_VALUE_STRING;
						  		tag.expr = GFA_String_Create((const char *)data, strlen((const char *)data));
						  		break;
						  	}
						  	
						  	default:
						  	  Assert(false);
						  }
					  }

						break;
					}
					
					// Nombres
					case GFA_TAG_BASE:
					  tag.tag = GFA_TAG_VALUE;
					case GFA_TAG_VALUE:
					{
					  switch (tag.subtag) {					  	
					  	case GFA_SUBTAG_INT_U8:
					  	{
					  		tag.info.number.value_U8 = *(unsigned char *)(file->ptr++);
					  		tag.info.number.value_float = (float)tag.info.number.value_U8;
					  		break;
					  	}
					  	
					  	case GFA_SUBTAG_INT_S16:
					  	{
					  	  tag.info.number.value_S16 = 0;
					  		tag.info.number.value_S16 |= (short)*(file->ptr++) << 8;
					  		tag.info.number.value_S16 |= (short)*(file->ptr++);
					  		tag.info.number.value_float = (float)tag.info.number.value_S16;
					  		break;
					  	}
					  	
					  	case GFA_SUBTAG_INT_S32:
					  	{
					  	  char *inumber = (char *)&tag.info.number.value_S32;
					  	  *(inumber++) = *(file->ptr++);
					  	  *(inumber++) = *(file->ptr++);
					  	  *(inumber++) = *(file->ptr++);
					  	  *inumber = *(file->ptr++);
					  	  tag.info.number.value_float = (float)tag.info.number.value_S32;
					  	  break;
					  	}
					  	
					  	case GFA_SUBTAG_FLOAT:
					  	{
					  	  memcpy(&tag.info.number.value_float, file->ptr, sizeof(float));
					  	  file->ptr += sizeof(float);
					  	  tag.info.number.value_S32 = (long)tag.info.number.value_float;
					  	  break;
					  	}
					  	
					  	case GFA_SUBTAG_STRING:
					  	{
					  		unsigned short size = (unsigned short)*(file->ptr++) << 8;
						    size |= *(file->ptr++);
						
						    tag.expr = GFA_String_Create(file->ptr, size);
						    file->ptr += size;
					  		break;
					  	}
					  	
					  	default:
					  	  return GFA_ERR_INTERNAL_UNKNOWN_SUBTAG;   
					  }
					  					    
					  break;
					}
					
				  // Erreur interne
          default:
            return GFA_ERR_INTERNAL_UNKNOWN_TAG;
				}
				
				// Affectation priorit de calcul
				/*if (tag.tag==GFA_TAG_MATH)
				  tag.priority = GFA_Math_PriorityTable[tag.subtag - 1];
				else if (tag.tag==GFA_TAG_LOGIC)
				  tag.priority = GFA_PRIORITY_LOGIC;  
				else if (tag.tag==GFA_TAG_COMPARE)
				  tag.priority = GFA_PRIORITY_COMPARE;
				*/
				
				// Ajout du tag dans la pile
				if (!GFA_Tag_PushFromStack(stack_tag, &tag, -1))
				  return GFA_ERR_STACK_TAG_OVERFLOW;
			}
		}
	}	
		
  return GFA_ERR_OK;
}
