/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2012 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#define _ISOC99_SOURCE
#include "bbowda.h"
#include "nlpopt.h"
#include "eval.h"
#include "gmmem.h"
#include "eqconst.h"
#include "covar.h"
#include "problem.h"
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <math.h>

static const int enable_global_search=true;

/* entry point of the BBOWDA algorithm */
void bbowda(const struct bbowda_problem *problem,
            const struct bbowda_params *params,
            void (*evaluate_F)(const struct bbowda_problem *problem,
                               const double *x, double *F, void *user_data),
            void *eval_user_data)
{
  size_t i;
  int ret;

  /* evaluate constraints at starting points */
  struct eval_points evalpts;
  init_points(&evalpts,problem,evaluate_F,eval_user_data);
  struct eqc_estimates *eqc=alloc_eqc_estimates();
  /* if we don't have enough starting points, generate more */
  size_t nstart;
  if (DIMX<5) {
    nstart = DIMX+DIMY+DIMY_EQ+6;
  } else if (DIMX<10) {
    nstart = DIMX+DIMX+DIMY+DIMY_EQ+2;
  } else {
    nstart = 4*DIMX+DIMY+DIMY_EQ-16;
  }
  if (evalpts.numcurrpts < nstart) {
    /* consider a grid of (nstart*2)*(nstart*2) points
       don't generate 2 points on the same xk */
    unsigned char used_x[DIMX][nstart*2];
    memset(used_x, 0, sizeof(used_x));
    /* mark all the already used xk */
    size_t i, k, n_used_x[DIMX];
    memset(n_used_x, 0, sizeof(n_used_x));
    double delta_x[DIMX];
    for (k=0; k<(size_t)DIMX; k++) {
      delta_x[k]=(problem->xup[k]-problem->xlow[k])/(double)(nstart*2-1);
    }
    for (i=0; i<evalpts.numcurrpts; i++) {
      /* round the given starting points to the closest x and y on the grid */
      double *x=currpts(&evalpts)[i];
      for (k=0; k<(size_t)DIMX; k++) {
        if (delta_x[k]!=0.) { /* filter out equality bounds */
          long j=lround((x[k]-problem->xlow[k])/delta_x[k]);
          if (j<0) { /* sanity check */
            printf("WARNING: Starting value %lf out of bounds [%lf,%lf].\n",
                   x[k],problem->xlow[k],problem->xup[k]);
            j=0;
          }
          if (j>=(long)(2*nstart)) { /* sanity check */
            printf("WARNING: Starting value %lf out of bounds [%lf,%lf].\n",
                   x[k],problem->xlow[k],problem->xup[k]);
            j=2*nstart-1;
          }
          if (!used_x[k][j]) n_used_x[k]++;
          used_x[k][j]=true;
        }
      }
    }
    /* now loop until we have enough starting points */
    while (evalpts.numcurrpts < nstart) {
      /* generate a starting point */
      /* try 10 points, pick the one the farthest away */
      double new_x[DIMX], dist=-INFINITY;
      size_t t, j[DIMX];
      for (t=0; t<10; t++) {
        double new_x_t[DIMX], dist_t=INFINITY;
        size_t j_t[DIMX], i;
        /* generate point */
        for (k=0; k<(size_t)DIMX; k++) {
          /* equiprobability among the not yet used xk */
          size_t j,jtk_index=((unsigned long long)rand()
                               *(unsigned long long)((size_t)(nstart*2)-n_used_x[k]))
                             /((unsigned long long)(RAND_MAX)+1ull);
          for (j=0; j<(size_t)(nstart*2); j++) {
            if (!used_x[k][j]) {
              if (!jtk_index--) {
                new_x_t[k]=problem->xlow[k]+(double)j*delta_x[k];
                j_t[k]=j;
                break;
              }
            }
          }
        }
        /* compute squared distance to the closest point */
        for (i=0; i<evalpts.numcurrpts; i++) {
          double *x=currpts(&evalpts)[i], dist_t_i=0.;
          for (k=0; k<(size_t)DIMX; k++) {
            dist_t_i+=(new_x_t[k]-x[k])*(new_x_t[k]-x[k]);
          }
          if (dist_t_i<dist_t) dist_t=dist_t_i;
        }
        /* keep the point if it's better */
        if (dist_t>dist) {
          dist=dist_t;
          memcpy(j,j_t,(size_t)DIMX*sizeof(size_t));
          memcpy(new_x,new_x_t,(size_t)DIMX*sizeof(double));
        }
      }
      /* mark the xk[j] as used */
      for (k=0; k<(size_t)DIMX; k++) {
        used_x[k][j[k]]=true;
        n_used_x[k]++;
      }
      /* evaluate constraints at the new starting point */
      printf("Generated starting point x=[");
      if (DIMX) {
        printf("%lf",new_x[0]);
      }
      for (i=1; i<(size_t)DIMX; i++) {
        printf(",%lf",new_x[i]);
      }
      printf("]\n");
      new_point(&evalpts,problem,
                params->global_search_ignores_eq_constraints?NULL:eqc,new_x,
                evaluate_F,eval_user_data);
    }
  }

  /* compute first global over-/underestimates of equality constraints */
  if (!params->global_search_ignores_eq_constraints) {
    compute_1st_global_eq_cst_estimates(eqc,problem,&evalpts);
  }
  const struct estimate_constraints *estimatec=eqc_get_estimatec(eqc);

  /* allocate vectors and matrices for the covariance models */
  struct covar *covar = alloc_covar(problem);

  /* optimize until we have reached the maximum number of evaluations */
  while (evalpts.numcurrpts < params->maxpts) {
    double optimum_x[DIMX];
    if (enable_global_search && (evalpts.numcurrpts & 1)) { /* global (gapfilling) search */
      /* compute a density function (gmm) */
      struct GMM *gmm=build_density_gmm(problem,&evalpts);
      /* minimize the density function under the constraints */
      if (estimatec->num_estimate_constraints) {
        ret=solve_nlp(1,&(struct GMM_problem){problem,gmm,estimatec,
                                              params->estimate_constraint_tol},
                      true,NULL,optimum_x);
        printf("Global presearch found x=[");
        if (DIMX) {
          printf("%lf",optimum_x[0]);
        }
        for (i=1; i<(size_t)DIMX; i++) {
          printf(",%lf",optimum_x[i]);
        }
        printf("] (reason: %d)\n",ret);
      }
      ret=solve_nlp(1,&(struct GMM_problem){problem,gmm,estimatec,
                                            params->estimate_constraint_tol},
                    !estimatec->num_estimate_constraints,NULL,optimum_x);
      /* free the density function parameters */
      free_gmm(gmm);
      printf("Global search found x=[");
    } else { /* local search */
      /* choose best point */
      get_best_point(&evalpts,problem);
      /* build regularized covariance model around the point */
      build_local_regcovar_model(covar,problem,&evalpts);
      /* optimize the covariance model */
      ret=solve_nlp(0,&(struct covar_problem){problem,covar},false,
                    evalpts.best_point,optimum_x);
      printf("Local search found x=[");
    }
    if (DIMX) {
      printf("%lf",optimum_x[0]);
    }
    for (i=1; i<(size_t)DIMX; i++) {
      printf(",%lf",optimum_x[i]);
    }
    printf("] (reason: %d)\n",ret);
    /* evaluate constraints at optimum */
    new_point(&evalpts,problem,
              params->global_search_ignores_eq_constraints?NULL:eqc,optimum_x,
              evaluate_F,eval_user_data);
  }

  /* free no longer needed data */
  free_covar(covar);
  free_eqc_estimates(eqc);

  if (DIMY_EQ) {
    extrapolate_point(&evalpts,problem,params->optimum_tol,evaluate_F,
                      eval_user_data);
  }

  double optimum_x[DIMX];
  printf("Optimum: %lf at x=[",get_optimum_x(&evalpts,problem,optimum_x,
                                             params->optimum_tol));
  if (DIMX) {
    printf("%lf",optimum_x[0]);
  }
  for (i=1; i<(size_t)DIMX; i++) {
    printf(",%lf",optimum_x[i]);
  }
  printf("]\n");

  free(evalpts.currpts_p);
}
