/* bbowdapp - C++ binding for bbowda
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include "bbowdapp.hh"

#include <cstring>

namespace Bbowda {
  SolverParameters::SolverParameters(size_t maxpts, double optimum_tol,
                                     bool global_search_ignores_eq_constraints,
                                     double estimate_constraint_tol)
  {
    bbowda_params::maxpts = maxpts;
    bbowda_params::optimum_tol = optimum_tol;
    bbowda_params::global_search_ignores_eq_constraints
      = global_search_ignores_eq_constraints;
    bbowda_params::estimate_constraint_tol = estimate_constraint_tol;
  }

  static const double * dvecdup(const double *orig, std::size_t dim)
  {
    if (!dim) {
      return static_cast<const double *>(0);
    }
    double *result = new double[dim];
    std::memcpy(result, orig, dim * sizeof(double));
    return result;
  }

  OptimizationProblem::OptimizationProblem(int dimx, int dimy, int dimy_eq,
                                           int numinitpts, const double *c,
                                           const double *xlow,
                                           const double *xup,
                                           const double *Flow,
                                           const double *Fup,
                                           const double *initpts_p,
                                           bool copyVectors)
  {
    bbowda_problem::dimx = dimx;
    bbowda_problem::dimy = dimy;
    bbowda_problem::dimy_eq = dimy_eq;
    bbowda_problem::numinitpts = numinitpts;

    if (copyVectors) {
      bbowda_problem::c = dvecdup(c, (std::size_t) dimx + dimy);
      bbowda_problem::xlow = dvecdup(xlow, dimx);
      bbowda_problem::xup = dvecdup(xup, dimx);
      bbowda_problem::Flow = dvecdup(Flow, dimy);
      bbowda_problem::Fup = dvecdup(Fup, dimy);
      bbowda_problem::initpts_p = dvecdup(initpts_p,
                                          (std::size_t) numinitpts * dimx);
    } else {
      bbowda_problem::c = c;
      bbowda_problem::xlow = xlow;
      bbowda_problem::xup = xup;
      bbowda_problem::Flow = Flow;
      bbowda_problem::Fup = Fup;
      bbowda_problem::initpts_p = initpts_p;
    }

    m_freeVectors = copyVectors;
  }

  OptimizationProblem::~OptimizationProblem()
  {
    if (m_freeVectors) {
      delete[] c;
      delete[] xlow;
      delete[] xup;
      delete[] Flow;
      delete[] Fup;
      delete[] initpts_p;
    }
  }

  // static
  void OptimizationProblem::evaluate_F_cb(const struct bbowda_problem *,
                                          const double *x, double *F,
                                          void *user_data) {
    static_cast<OptimizationProblem *>(user_data)->evaluateF(x, F);
  }

  void OptimizationProblem::solve(const SolverParameters &params) {
    bbowda(this, &params, evaluate_F_cb, this);
  }
}
