/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2024 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include "bbowdapp.hh"
#include <cstdlib>

// USER INPUT: Problem dimensions
#define DIM_X 2 // scale this to scale the problem
#define DIM_Y 1
#define DIM_Y_EQ 0

// USER INPUT: Number of starting points
#define NUM_INIT_PTS 0

// USER INPUT:
static struct ProblemData {
  // min cT (x, F(x))
  double c_data[DIM_X+DIM_Y];
  // s.t. Flow <= F(x) <= Fup
  static const double Flow_data[DIM_Y];
  static const double Fup_data[DIM_Y];
  //      xlow <= x <= xup
  double xlow_data[DIM_X];
  double xup_data[DIM_X];
  // starting points
#if NUM_INIT_PTS > 0
#if __cplusplus < 201103L
  static const double initpts_data[NUM_INIT_PTS * DIM_X];
#else // we have the delegating constructor, so we can write:
  static const double initpts_data[NUM_INIT_PTS][DIM_X];
#endif
#endif

  ProblemData() {
    // initialize data that cannot be static const
    for (int i = 0; i < DIM_X; i++) c_data[i] = 0.;
    c_data[DIM_X] = 1.;
    for (int i = 0; i < DIM_X; i++) xlow_data[i] = -1.;
    for (int i = 0; i < DIM_X; i++) xup_data[i] = 2.;
  }
} problem_data;

// initialize static const data
const double ProblemData::Flow_data[] = {-1000.};
const double ProblemData::Fup_data[] = {2700.};
#if NUM_INIT_PTS > 0
#if __cplusplus < 201103L
const double ProblemData::initpts_data[NUM_INIT_PTS * DIM_X]={};
#else // we have the delegating constructor, so we can write:
const double ProblemData::initpts_data[NUM_INIT_PTS][DIM_X]={};
#endif
#endif

#if __cplusplus < 201103L
#define override /**/
#define nullptr 0
#endif

class Problem : public Bbowda::OptimizationProblem {
  public:
    Problem() : Bbowda::OptimizationProblem(DIM_X, DIM_Y, DIM_Y_EQ,
                                            NUM_INIT_PTS, problem_data.c_data,
                                            problem_data.xlow_data,
                                            problem_data.xup_data,
                                            problem_data.Flow_data,
                                            problem_data.Fup_data,
#if NUM_INIT_PTS > 0
                                            problem_data.initpts_data,
#else
                                            nullptr,                                      
#endif
                                            false) {}
  protected:
    // evaluate F(x)
    void evaluateF(const double *x, double *F) override
    {
      int i;
      *F = 0.;
      for (i = 0; i < dimx - 1; i++)
        *F += (1. - x[i]) * (1. - x[i]) + 100. * (x[i+1] - x[i]*x[i]) * (x[i+1] - x[i]*x[i]);
    }
};

// main function
int main(void) {
  // ensure reproducible results
  std::srand(31337);

  // initialize and/or allocate problem, if necessary
  Problem problem;

  // run the optimization
  problem.solve(Bbowda::SolverParameters(
    // USER INPUT: Maximum points to evaluate
    /* maxpts = */ 200,
    // USER INPUT: Tolerance for optimum feasibility
    /* optimum_tol = */ .001,
    // USER INPUT: If non-zero, ignore equality constraints for global search
    /* global_search_ignores_eq_constraints = */ false,
    /* USER INPUT: Tolerance for the constraints estimating the equality constraints
                   during global search */
    /* estimate_constraint_tol = */ .01
  ));

  return 0;
}
