// kate: hl c++
// This is a SWIG input file, consisting of C++ code and SWIG directives.

/* bbowdapp - C++ binding for bbowda
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

// generate directors for all classes that have virtual methods
%feature("director");

%{
#include <bbowdapp.hh>
#include <cstdlib>
%}

%include <std_common.i>

#ifdef SWIGJAVA
%include <arrays_java.i>
%rename("%(lowercamelcase)s", %$isvariable) "";
%typemap(in) const double[ANY] (double *jarr) %{
  if ($input) {
    if (!SWIG_JavaArrayInDouble(jenv, &jarr, (double **)&$1, $input)) return $null;
  } else {
    $1 = static_cast<double *>(0);
  }
  %}
%typemap(argout) const double[ANY]
%{ if ($input) jenv->ReleaseDoubleArrayElements($input, jarr$argnum, JNI_ABORT); %}
%typemap(javadirectorin) double[ANY] "$jniinput"
%typemap(directorin,descriptor="[D") double[ANY] %{
#define arg1 this
  $input = jenv->NewDoubleArray($1_dim0);
  jenv->SetDoubleArrayRegion($input, 0, $1_dim0, $1);
#undef arg1
%}
%typemap(directorargout) const double[ANY] %{%}
%typemap(directorargout) double[ANY] %{
#define arg1 this
  jenv->GetDoubleArrayRegion($input, 0, $1_dim0, $1);
#undef arg1
%}
typedef const double bbowda_swig_x_t[arg1->dimx];
typedef double bbowda_swig_F_t[arg1->dimy];
#endif

#ifdef SWIGPYTHON
%{
#ifdef _WIN32
#include <malloc.h>
#ifndef alloca
#define alloca _alloca
#endif
#else
#include <alloca.h>
#endif
%}
%include <carrays.i>
%ignore DoubleArray::cast;
%ignore DoubleArray::frompointer;
%feature("docstring", "Array of double, wrapping a C double[].") DoubleArray;
%array_class(double,DoubleArray)
%typemap(in) const double[] (void *argp = 0, int res = 0) %{
  res = SWIG_ConvertPtr($input, &argp, SWIGTYPE_p_DoubleArray, 0);
  if (SWIG_IsOK(res)) { // wrapped C pointer
    arg$argnum = reinterpret_cast< double * >(argp);
  } else if (PySequence_Check($input)) { // Python (or NumPy) sequence
    Py_ssize_t n = PySequence_Length($input);
    if (n < 0) {
      SWIG_exception_fail(SWIG_ValueError, "PySequence_Length failed, in "
        "method '$symname', argument $argnum of type 'const double []'");
    }
    double *arg = reinterpret_cast<double *>(alloca(n * sizeof(double)));
    for (Py_ssize_t i = 0; i < n; i++) {
      PyObject *o = PySequence_GetItem($input, i);
      if (PyNumber_Check(o)) {
        arg[i] = (double) PyFloat_AsDouble(o);
      } else {
        SWIG_exception_fail(SWIG_ValueError, "Sequence element not a number, "
          "in method '$symname', argument $argnum of type 'const double []'");
      }
    }
    arg$argnum = arg;
  } else {
    SWIG_exception_fail(SWIG_ArgError(res), "in method '$symname', argument "
      "$argnum of type 'const double []'");
  }
%}
%typemap(in) const double initpts[] (void *argp = 0, int res = 0) %{
  // initpts argument, may be a 2D sequence or a flattened 1D sequence
  res = SWIG_ConvertPtr($input, &argp, SWIGTYPE_p_DoubleArray, 0);
  if (SWIG_IsOK(res)) { // wrapped C pointer
    arg$argnum = reinterpret_cast< double * >(argp);
  } else if (PySequence_Check($input)) { // Python (or NumPy) sequence
    Py_ssize_t n = PySequence_Length($input);
    if (n < 0) {
      SWIG_exception_fail(SWIG_ValueError, "PySequence_Length failed, in "
        "method '$symname', argument $argnum of type 'const double []'");
    }
    double *arg;
    if ((n > 0) && PySequence_Check(PySequence_GetItem($input, 0))) {
      // 2D array, flatten
      Py_ssize_t m = PySequence_Length(PySequence_GetItem($input, 0));
      if (m < 0) {
        SWIG_exception_fail(SWIG_ValueError, "PySequence_Length failed, in "
          "method '$symname', argument $argnum of type 'const double []'");
      }
      arg = reinterpret_cast<double *>(alloca(n * m * sizeof(double)));
      for (Py_ssize_t i = 0; i < n; i++) {
        PyObject *o = PySequence_GetItem($input, i);
        if (PySequence_Check(o)) {
          for (Py_ssize_t j = 0; j < m; j++) {
            PyObject *p = PySequence_GetItem(o, i);
            if (PyNumber_Check(p)) {
              arg[i * m + j] = (double) PyFloat_AsDouble(p);
            } else {
              SWIG_exception_fail(SWIG_ValueError, "Sequence element not a "
                "number, in method '$symname', argument $argnum of type 'const "
                "double []'");
            }
          }
        } else {
          SWIG_exception_fail(SWIG_ValueError, "Sequence element not a "
            "sequence, in method '$symname', argument $argnum of type 'const "
            "double []'");
        }
      }
    } else {
      // 1D array, already flattened
      arg = reinterpret_cast<double *>(alloca(n * sizeof(double)));
      for (Py_ssize_t i = 0; i < n; i++) {
        PyObject *o = PySequence_GetItem($input,i);
        if (PyNumber_Check(o)) {
          arg[i] = (double) PyFloat_AsDouble(o);
        } else {
          SWIG_exception_fail(SWIG_ValueError, "Sequence element not a number, "
            "in method '$symname', argument $argnum of type 'const double []'");
        }
      }
    }
    arg$argnum = arg;
  } else {
    SWIG_exception_fail(SWIG_ArgError(res), "in method '$symname', argument "
      "$argnum of type 'const double []'");
  }
%}
%typemap(in) double[] (void *argp = 0, int res = 0) %{
  res = SWIG_ConvertPtr($input, &argp, SWIGTYPE_p_DoubleArray, 0);
  if (!SWIG_IsOK(res)) {
    SWIG_exception_fail(SWIG_ArgError(res), "in method '$symname', argument "
      "$argnum of type 'double []'");
  }
  arg$argnum = reinterpret_cast< double * >(argp);
%}
%typemap(directorin) double[] %{
  $input = SWIG_NewPointerObj(SWIG_as_voidptr($1), SWIGTYPE_p_DoubleArray, 0);
%}
%typemap(out) const double[ANY] {
  %set_output(SWIG_NewPointerObj(SWIG_as_voidptr($1), SWIGTYPE_p_DoubleArray,
                                 0));
}
typedef const double bbowda_swig_x_t[];
typedef double bbowda_swig_F_t[];
#endif

#include <cstddef>

%{
  typedef const double bbowda_swig_x_t[];
  typedef double bbowda_swig_F_t[];
%}

namespace Bbowda {
  /**
   * @brief Solver parameters for the BBOWDA algorithm.
   *
   * Parameters allowing to tune how the BBOWDA algorithm operates.
   */
  class SolverParameters {
    public:
      /**
       * @brief Constructor.
       *
       * @param maxpts Maximum number of points to evaluate. See @ref maxpts.
       * @param optimum_tol Tolerance for optimum feasibility (infinity norm).
       *                    See @ref optimum_tol.
       * @param global_search_ignores_eq_constraints Whether to ignore implicit
       *   equality constraints for global search. See
       *   @ref global_search_ignores_eq_constraints.
       * @param estimate_constraint_tol Tolerance for the constraints estimating
       *                                the implicit equality constraints during
       *                                global search. See
       *                                @ref estimate_constraint_tol.
       */
      SolverParameters(size_t maxpts, double optimum_tol,
                       bool global_search_ignores_eq_constraints,
                       double estimate_constraint_tol);

      /**
       * @brief Maximum number of points to evaluate.
       *
       * Currently, the algorithm will always run until exactly this many points
       * are evaluated, because no other stopping criteria are implemented yet.
       * In the presence of implicit equality constraints, it may then evaluate
       * one more point for the final extrapolation attempt.
       */
      std::size_t maxpts;

      /**
       * @brief Tolerance for optimum feasibility (infinity norm).
       *
       * A point will be considered feasible, and thus a valid candidate for the
       * optimum, if the bound constraints for @a x are satisfied exactly, and
       * if none of the other constraints is violated by more than
       * @a optimum_tol, i.e., if the componentwise inequalities @a Flow -
       * @a optimum_tol <= F1(@a x) <= @a Fup + @a optimum_tol and -
       * @a optimum_tol < F2(@a x) < @a optimum_tol hold. In vector terms, this
       * means that the infinity norm of the constraint violation must be less
       * than @a optimum_tol.
       */
      double optimum_tol;

      /**
       * @brief Whether to ignore implicit equality constraints for global
       * search.
       *
       * True means that implicit equality constraints will be ignored by the
       * global search. Hence, it will suggest evaluation points to fill any
       * gaps in the search space even if they are nowhere near feasible for the
       * implicit equality constraints.
       *
       * False means that implicit equality constraints will be honored by the
       * global search. The global search will compute estimates that attempt to
       * globally enclose the implicit equality constraints with the help of an
       * external LP solver, helping to suggest only points that are expected to
       * be approximately feasible.
       *
       * This parameter has no effect if the problem does not include any
       * implicit equality constraints.
       */
      bool global_search_ignores_eq_constraints;

      /**
       * @brief Tolerance for the constraints estimating the implicit equality
       *        constraints during global search.
       *
       * If @ref global_search_ignores_eq_constraints is false, the global
       * search attempts to estimate global enclosures for the implicit equality
       * constraints. But those enclosures are estimated approximations and not
       * guaranteed to hold exactly. This tolerance specifies by how much the
       * bounds for the estimated enclosures should be relaxed to account for
       * that.
       *
       * This parameter has no effect if the problem does not include any
       * implicit equality constraints or if
       * @ref global_search_ignores_eq_constraints is true.
       */
      double estimate_constraint_tol;
  };

  /**
   * @brief Definition of a black box optimization problem.
   *
   * This class represents both the constants (inherited from
   * @ref bbowda_problem) and the black box function (in the form of the pure
   * virtual method @ref evaluateF) that together represent the black box
   * optimization problem. Subclass this class to implement your optimization
   * problem.
   *
   * The problem is assumed to be of the form:
   * @code{.unparsed}
   * min cT (x ; y)
   * s.t. y = F1(x) [explicit equality constraints]
   *      F2(x) = 0 [implicit equality constraints]
   *      xlow <= x <= xup
   *      Flow <= y <= Fup
   * @endcode
   */
  class OptimizationProblem {
    public:
      /**
       * @brief Main constructor.
       *
       * @param dimx Number of input variables. See @ref dimx.
       * @param dimy Number of explicit equality constraints. See @ref dimy.
       * @param dimy_eq Number of implicit equality constraints. See
       *                @ref dimy_eq.
       * @param numinitpts Number of user-specified starting points. See
       *                   @ref numinitpts.
       * @param c Coefficients in the (linear) objective function. See @ref c.
       * @param xlow Lower bounds for input variables. See @ref xlow.
       * @param xup Upper bounds for input variables. See @ref xup.
       * @param Flow Lower bounds for explicit equality constraints. See
       *             @ref Flow.
       * @param Fup Upper bounds for explicit equality constraints. See
       *            @ref Fup.
       * @param initpts User-provided starting points. See @ref initpts.
       * @param copyVectors Whether to copy the vectors.
       */
      // The optional parameter copyVectors is not wrapped, it is always true in
      // the bindings.
      OptimizationProblem(int dimx, int dimy, int dimy_eq, int numinitpts,
                          const double c[], const double xlow[],
                          const double xup[], const double Flow[],
                          const double Fup[], const double initpts[]);

      /**
       * @brief Virtual destructor.
       */
      virtual ~OptimizationProblem();

      /**
       * @brief Main entry point of the BBOWDA object-oriented API.
       *
       * Runs the BBOWDA algorithm on this problem with the given parameters.
       *
       * @param params The solver parameters for the BBOWDA algorithm.
       */
      void solve(const SolverParameters &params);

      /**
       * @brief Number of input variables.
       *
       * The vector dimension of @a x.
       */
      const int dimx;

      /**
       * @brief Number of explicit equality constraints.
       *
       * The vector dimension of @a y = F1(@a x). Also known as the number of
       * interval inequality constraints if the coefficients of @a y in @ref c
       * are 0.
       */
      const int dimy;

      /**
       * @brief Number of implicit equality constraints.
       *
       * Vector dimension of F2(@a x).
       */
      const int dimy_eq;

      /**
       * @brief Number of user-specified starting points.
       *
       * Can be 0. If no or not enough starting points are provided by the user,
       * BBOWDA will automatically sample some starting points within the
       * bounds.
       */
      const int numinitpts;


      /**
       * @brief Coefficients in the (linear) objective function.
       *
       * The coefficient vector @a c. Must have dimension @ref dimx + @ref dimy.
       * First list all coefficients of @a x in order, then all coefficients of
       * @a y = F1(@a x) in order.
       */
      const double c[arg1->dimx+arg1->dimy];

      /**
       * @brief Lower bounds for input variables.
       *
       * Lower bounds for @a x. Must have dimension @ref dimx.
       */
      const double xlow[arg1->dimx];

      /**
       * @brief Upper bounds for input variables.
       *
       * Upper bounds for @a x. Must have dimension @ref dimx.
       */
      const double xup[arg1->dimx];

      /**
       * @brief Lower bounds for explicit equality constraints.
       *
       * Lower bounds for @a y = F1(@a x). Must have dimension @ref dimy.
       *
       * @note No bounds need to be specified for the implicit equality
       * constraints F2(@a x) because those bounds are by definition always 0.
       */
      const double Flow[arg1->dimy];

      /**
       * @brief Upper bounds for explicit equality constraints.
       *
       * Upper bounds for @a y = F1(@a x). Must have dimension @ref dimy.
       *
       * @note No bounds need to be specified for the implicit equality
       * constraints F2(@a x) because those bounds are by definition always 0.
       */
      const double Fup[arg1->dimy];

      %rename(initptsvec) initpts_p;
      /**
       * @brief User-provided starting points, as a contiguous array.
       *
       * Must be a contiguous matrix of dimension @ref numinitpts * @ref dimx
       * (in row-major order, i.e., first all components of the first starting
       * point, then the second one, etc.). If @ref numinitpts is 0, this is
       * just an empty vector (and can be NULL).
       */
      const double initpts_p[arg1->numinitpts*arg1->dimx];

    protected:
      /**
       * @brief Pure virtual callback evaluating the black box function.
       *
       * Callback evaluating the black box function (or obtaining the evaluation
       * result from an external source) at the point @a x (of dimension
       * @ref dimx) and writing the result to @a F (of dimension @ref dimy +
       * @ref dimy_eq). Must be implemented by the user through subclassing.
       *
       * @note There is no @a user_data pointer because any user data can and
       * should be included in the user-provided subclass.
       */
      virtual void evaluateF(bbowda_swig_x_t x, bbowda_swig_F_t F) = 0;
  };
}

// wrap this cstdlib function because BBOWDA uses the C library's random seed,
// not the Python or Java ones
namespace std {
  /**
   * @brief Set the pseudorandom number generator (PRNG) seed of the C library.
   *
   * BBOWDA uses the C library to generate pseudorandom numbers. Therefore, if
   * reproducible results are needed, it is necessary to set this seed.
   *
   * @param seed The random seed to set.
   */
  void srand(unsigned int seed);
}

#ifdef SWIGJAVA
%extend Bbowda::OptimizationProblem {
  %proxycode %{
    private static double[] flatten2DArray(double[][] arr) {
      if (arr == null) {
        return null;
      }

      if (arr.length == 0) {
        return new double[0];
      }

      int n = arr.length;
      int m = arr[0].length;
      double[] flat = new double[n * m];
      for (int i = 0; i < n; i++) {
        System.arraycopy(arr[i], 0, flat, i * m, m);
      }
      return flat;
    }

    /**
     * Convenience constructor.<br>
     * <br>
     * This constructor overload is syntactic sugar allowing to pass a<br>
     * 2-dimensional array as <i>initpts</i>.<br>
     * <br>
     * @param dimx Number of input variables. See <a href="#dimx">dimx</a>.<br>
     * @param dimy Number of explicit equality constraints. See
     * <a href="#dimy">dimy</a>.<br>
     * @param dimy_eq Number of implicit equality constraints. See<br>
     *                <a href="#dimy_eq">dimy_eq</a>.<br>
     * @param numinitpts Number of user-specified starting points. See<br>
     *                   <a href="#numinitpts">numinitpts</a>.<br>
     * @param c Coefficients in the (linear) objective function. See
     * <a href="#c">c</a>.<br>
     * @param xlow Lower bounds for input variables. See
     * <a href="#xlow">xlow</a>.<br>
     * @param xup Upper bounds for input variables. See
     * <a href="#xup">xup</a>.<br>
     * @param Flow Lower bounds for explicit equality constraints. See<br>
     *             <a href="#Flow">Flow</a>.<br>
     * @param Fup Upper bounds for explicit equality constraints. See<br>
     *            <a href="#Fup">Fup</a>.<br>
     * @param initpts User-provided starting points. See
     * <a href="#initpts">initpts</a>.<br>
     */
    public OptimizationProblem(int dimx, int dimy, int dimy_eq, int numinitpts,
                               double[] c, double[] xlow, double[] xup,
                               double[] Flow, double[] Fup,
                               double[][] initpts) {
      this(dimx, dimy, dimy_eq, numinitpts, c, xlow, xup, Flow, Fup,
           flatten2DArray(initpts));
    }

    /**
     * User-provided starting points, as a 2-dimensional jagged array.<br>
     * <br>
     * A jagged matrix of dimension <a href="#numinitpts">numinitpts</a> *
     * <a href="#dimx">dimx</a><br>
     * (in row-major order, i.e., first all components of the first starting<br>
     * point, then the second one, etc.). If
     * <a href="#numinitpts">numinitpts</a> is 0, this array is empty.
     */
    public double[][] getInitpts() {
      double[] initptsvec = getInitptsvec();
      int numinitpts = getNuminitpts();
      int dimx = getDimx();
      double[][] initpts = new double[numinitpts][dimx];
      for (int i = 0; i < numinitpts; i++) {
        System.arraycopy(initptsvec, i * dimx, initpts[i], 0, dimx);
      }
      return initpts;
    }
  %}
}
#endif

#ifdef SWIGPYTHON
%pythoncode %{
class OptimizationProblem(OptimizationProblem):
  __doc__ = OptimizationProblem.__doc__

  # check that evaluateF is properly overridden by the subclass
  def __init__(self, dimx, dimy, dimy_eq, numinitpts, c, xlow, xup, Flow, Fup,
               initpts):
    super().__init__(dimx, dimy, dimy_eq, numinitpts, c, xlow, xup, Flow, Fup,
                     initpts)

    if type(self).evaluateF is OptimizationProblem.evaluateF:
      raise TypeError("".join(("Can't instantiate abstract class ",
                               type(self).__name__,
                               " with abstract method evaluateF")))

  # unflatten initptsvec to a 2-dimensional list
  def initpts(self):
    """
User-provided starting points, as a list of lists.

A jagged matrix of dimension 'numinitpts' * 'dimx' (in row-major order, i.e.,
first all components of the first starting point, then the second one, etc.).
If 'numinitpts' is 0, this is just an empty list.
"""
    initptsvec = self.initptsvec
    numinitpts = self.numinitpts
    dimx = self.dimx
    # slice is not supported on the C array, so do it manually
    return [[initptsvec[j] for j in range(i*dimx, (i+1)*dimx)]
            for i in range(numinitpts)]

  initpts = property(initpts)
%}
#endif
