/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2012 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#define _ISOC99_SOURCE
#include "eval.h"
#include "problem.h"
#include "eqconst.h"
#include "xmalloc.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>

void init_points(struct eval_points* evalpts,
                 const struct bbowda_problem *problem,
                 void (*evaluate_F)(const struct bbowda_problem *problem,
                                    const double *x, double *F,
                                    void *user_data),
                 void *eval_user_data)
{
  size_t i;
  evalpts->numcurrpts=problem->numinitpts;
  evalpts->currpts_p=xmalloc((size_t)problem->numinitpts*((size_t)(DIMX+DIMY+DIMY_EQ))*sizeof(double));
  for (i=0; i<(size_t)problem->numinitpts; i++) {
    memcpy(currpts(evalpts)[i],initpts(problem)[i],DIMX*sizeof(double));
    evaluate_F(problem,initpts(problem)[i],currpts(evalpts)[i]+DIMX,
               eval_user_data);
  }
}

void new_point(struct eval_points* evalpts,
               const struct bbowda_problem *problem, struct eqc_estimates *eqc,
               const double *x,
               void (*evaluate_F)(const struct bbowda_problem *problem,
                                  const double *x, double *F, void *user_data),
               void *eval_user_data)
{
  size_t i=evalpts->numcurrpts++;
  evalpts->currpts_p=xrealloc(evalpts->currpts_p,evalpts->numcurrpts*((size_t)(DIMX+DIMY+DIMY_EQ))*sizeof(double));
  memcpy(currpts(evalpts)[i],x,DIMX*sizeof(double));
  evaluate_F(problem,x,currpts(evalpts)[i]+DIMX,eval_user_data);
  /* compute global over-/underestimates of equality constraints around the new point */
  if (eqc) {
    compute_global_eq_cst_estimates_around(eqc,problem,evalpts,
                                           currpts(evalpts)[i]);
  }
}

double get_optimum_x(struct eval_points* evalpts,
                     const struct bbowda_problem *problem, double *optimum_x,
                     double tol)
{
  size_t j;
  double optimum=INFINITY;
  for (j=0; j<evalpts->numcurrpts; j++) {
    double cTx=0.;
    double *x=currpts(evalpts)[j];
    size_t i;
    for (i=0; i<(size_t)DIMX; i++) {
      if (x[i]<problem->xlow[i] || x[i]>problem->xup[i]) {
        goto infeasible; /* infeasible point */
      }
    }
    for (i=0; i<(size_t)DIMY; i++) {
      if (x[DIMX+i]<problem->Flow[i]-tol || x[DIMX+i]>problem->Fup[i]+tol) {
        goto infeasible; /* infeasible point */
      }
    }
    for (i=0; i<(size_t)DIMY_EQ; i++) {
      if (fabs(x[DIMX+DIMY+i])>=tol) {
        goto infeasible; /* infeasible point */
      }
    }
    for (i=0; i<(size_t)(DIMX+DIMY); i++) {
      cTx+=problem->c[i]*x[i];
    }
    if (cTx<optimum) {
      memcpy(optimum_x,x,(size_t)DIMX*sizeof(double));
      optimum=cTx;
    }
    infeasible:;
  }
  return optimum;
}

/* The weights are determined using a penalty function approach, with a penalty
   term increasing over time:
   min cT x + evalpts->numcurrpts * constraint_violation */
static double get_point_penalty(struct eval_points* evalpts,
                                const struct bbowda_problem *problem,
                                const double *x)
{
  double penalty=0.;
  size_t i;
  for (i=0; i<(size_t)DIMX; i++) {
    if (x[i]<problem->xlow[i]) {
      penalty+=evalpts->numcurrpts*(problem->xlow[i]-x[i]);
    } else if (x[i]>problem->xup[i]) {
      penalty+=evalpts->numcurrpts*(x[i]-problem->xup[i]);
    }
  }
  for (i=0; i<(size_t)DIMY; i++) {
    if (x[DIMX+i]<problem->Flow[i]) {
      penalty+=evalpts->numcurrpts*(problem->Flow[i]-x[DIMX+i]);
    } else if (x[DIMX+i]>problem->Fup[i]) {
      penalty+=evalpts->numcurrpts*(x[DIMX+i]-problem->Fup[i]);
    }
  }
  for (i=0; i<(size_t)DIMY_EQ; i++) {
    penalty+=(evalpts->numcurrpts<18?sqrt(evalpts->numcurrpts/18.):
                  evalpts->numcurrpts<77?evalpts->numcurrpts/18.:
                  evalpts->numcurrpts*sqrt(evalpts->numcurrpts)/158.)*fabs(x[DIMX+DIMY+i]);
  }
  return penalty;
}

static double get_point_constraint_violation(
  const struct bbowda_problem *problem, const double *x)
{
  double penalty=0.;
  size_t i;
  for (i=0; i<(size_t)DIMX; i++) {
    if (x[i]<problem->xlow[i]) {
      penalty+=(problem->xlow[i]-x[i]);
    } else if (x[i]>problem->xup[i]) {
      penalty+=(x[i]-problem->xup[i]);
    }
  }
  for (i=0; i<(size_t)DIMY; i++) {
    if (x[DIMX+i]<problem->Flow[i]) {
      penalty+=(problem->Flow[i]-x[DIMX+i]);
    } else if (x[DIMX+i]>problem->Fup[i]) {
      penalty+=(x[DIMX+i]-problem->Fup[i]);
    }
  }
  for (i=0; i<(size_t)DIMY_EQ; i++) {
    penalty+=fabs(x[DIMX+DIMY+i]);
  }
  return penalty;
}

static double get_point_cTx(const struct bbowda_problem *problem,
                            const double *x)
{
  double cTx=0.;
  size_t i;
  for (i=0; i<(size_t)(DIMX+DIMY); i++) {
    cTx+=problem->c[i]*x[i];
  }
  return cTx;
}

double get_point_goodness(struct eval_points* evalpts,
                          const struct bbowda_problem *problem,
                          const double *x)
{
  return get_point_cTx(problem,x)+get_point_penalty(evalpts,problem,x);
}

void get_best_point(struct eval_points* evalpts,
                    const struct bbowda_problem *problem)
{
#ifdef USE_PENALTY_APPROACH
  size_t j;
  double optimum=INFINITY;
  best_point=NULL;
  for (j=0; j<evalpts->numcurrpts; j++) {
    double cTx_penalty=get_point_goodness(currpts(evalpts)[j]);
    if (cTx_penalty<optimum) {
      best_point=currpts(evalpts)[j];
      optimum=cTx_penalty;
    }
  }
  if (!best_point) best_point=currpts(evalpts)[0];
  best_goodness=optimum;
#else /* filter method */
  static unsigned char *used=NULL;
  static size_t used_size=0;
  /* use the penalty to compute the optimum goodness, it's used for weighting */
  double cTx[evalpts->numcurrpts], penalty[evalpts->numcurrpts];
  size_t i,j;
  double optimum=INFINITY;
  evalpts->best_point=NULL;
  /* new points are not used */
  used=xrealloc(used,evalpts->numcurrpts);
  memset(used+used_size,0,evalpts->numcurrpts-used_size);
  used_size=evalpts->numcurrpts;
  for (j=0; j<evalpts->numcurrpts; j++) {
    double cTx_j=get_point_cTx(problem,currpts(evalpts)[j]);
    double penalty_j=get_point_penalty(evalpts,problem,currpts(evalpts)[j]);
    double cTx_penalty=cTx_j+penalty_j;
    cTx[j]=cTx_j;
    penalty[j]=penalty_j;
    if (cTx_penalty<optimum) {
      optimum=cTx_penalty;
    }
  }
  evalpts->best_goodness=optimum;
  /* now pick a "best point" using the filter approach */
  unsigned char pareto_optimal[evalpts->numcurrpts];
  size_t num_pareto_points=0;
  memset(pareto_optimal,1,evalpts->numcurrpts);
  for (i=0; i<evalpts->numcurrpts; i++) {
    for (j=0; j<evalpts->numcurrpts; j++) {
      /* if j is strictly better than i, i is not Pareto-optimal */
      if (j!=i && pareto_optimal[j] /* avoid unnecessary comparisons */ &&
          cTx[j] <= cTx[i] && penalty[j] <= penalty[i] &&
          (cTx[j] < cTx[i] || penalty[j] < penalty[i])) {
        pareto_optimal[i]=0;
        goto dont_count;
      }
    }
    num_pareto_points++;
    dont_count:;
  }
  /* pick a pseudorandom Pareto-optimal point as the best point
     pick a later one with a higher probability */
  /* find the most recent Pareto-optimal point */
  /* i is unsigned and will overflow to ULONG_MAX */
  for (i=evalpts->numcurrpts-1; i<evalpts->numcurrpts; i--) {
    if (pareto_optimal[i]) break;
  }
  /* if it is not used, pick it with a probability of 1-1/num_pareto_points */
  if (used[i] || rand()<(RAND_MAX/(int)num_pareto_points)) {
    /* otherwise use equiprobability */
    size_t pareto_point_index=((unsigned long long)rand()
                                *(unsigned long long)num_pareto_points)
                              /((unsigned long long)(RAND_MAX)+1ull);
    for (i=0; i<evalpts->numcurrpts; i++) {
      if (pareto_optimal[i]) {
        if (!pareto_point_index--) break;
      }
    }
  }
  evalpts->best_point=currpts(evalpts)[i];
  used[i]=1;
  printf("Starting local search at x=[");
  if (DIMX) {
    printf("%lf",evalpts->best_point[0]);
  }
  for (i=1; i<(size_t)DIMX; i++) {
    printf(",%lf",evalpts->best_point[i]);
  }
  printf("]\n");
  return;
#endif
}

void extrapolate_point(struct eval_points* evalpts,
                       const struct bbowda_problem *problem, double optimum_tol,
                       void (*evaluate_F)(const struct bbowda_problem *problem,
                                          const double *x, double *F,
                                          void *user_data),
                       void *eval_user_data)
{
  double cTx[evalpts->numcurrpts], penalty[evalpts->numcurrpts];
  size_t i,j;
  for (j=0; j<evalpts->numcurrpts; j++) {
    cTx[j]=get_point_cTx(problem,currpts(evalpts)[j]);
    penalty[j]=get_point_constraint_violation(problem,currpts(evalpts)[j]);
  }
  /* determine the Pareto-optimal points */
  unsigned char pareto_optimal[evalpts->numcurrpts];
  size_t num_pareto_points=0;
  memset(pareto_optimal,1,evalpts->numcurrpts);
  for (i=0; i<evalpts->numcurrpts; i++) {
    for (j=0; j<evalpts->numcurrpts; j++) {
      /* if j is strictly better than i, i is not Pareto-optimal */
      if (j!=i && pareto_optimal[j] /* avoid unnecessary comparisons */ &&
          cTx[j] <= cTx[i] && penalty[j] <= penalty[i] &&
          (cTx[j] < cTx[i] || penalty[j] < penalty[i])) {
        pareto_optimal[i]=0;
        goto dont_count;
      }
    }
    num_pareto_points++;
    dont_count:;
  }
  /* throw away the points with excess constraint violation
     also throw away the points which are already feasible up to the tolerance,
     it's no use getting any better there */
  for (i=0; i<evalpts->numcurrpts; i++) {
    if (pareto_optimal[i]
        && (penalty[i]>optimum_tol*256. || penalty[i]<optimum_tol)) {
      pareto_optimal[i]=0;
      num_pareto_points--;
    }
  }
  /* do a ratio reject to get rid of outliers */
  if (num_pareto_points>=3) {
    double rho[evalpts->numcurrpts],rho_mean=0.,rho_var=0.;
    for (i=0; i<evalpts->numcurrpts; i++) {
      if (pareto_optimal[i]) {
        /* compute rho[i] */
        double d_i_NNi=INFINITY, d_NNi_NNNNi=INFINITY;
        size_t NNi;
        /* find nearest neighbor of i and distance */
        for (j=0; j<evalpts->numcurrpts; j++) {
          if (j!=i && pareto_optimal[j]) {
            double d_i_j=0;
            size_t k;
            for (k=0; k<(size_t)DIMX; k++) {
              d_i_j+=(currpts(evalpts)[j][k]-currpts(evalpts)[i][k])*(currpts(evalpts)[j][k]-currpts(evalpts)[i][k]);
            }
            if (d_i_j<d_i_NNi) {
              d_i_NNi=d_i_j;
              NNi=j;
            }
          }
        }
        /* find nearest neighbor of NNi and distance */
        for (j=0; j<evalpts->numcurrpts; j++) {
          if (j!=i && j!=NNi && pareto_optimal[j]) {
            double d_NNi_j=0;
            size_t k;
            for (k=0; k<(size_t)DIMX; k++) {
              d_NNi_j+=(currpts(evalpts)[j][k]-currpts(evalpts)[NNi][k])*(currpts(evalpts)[j][k]-currpts(evalpts)[NNi][k]);
            }
            if (d_NNi_j<d_NNi_NNNNi) {
              d_NNi_NNNNi=d_NNi_j;
            }
          }
        }
        /* rho[i] is the ratio */
        rho[i]=d_i_NNi/d_NNi_NNNNi;
        rho_mean+=rho[i];
      }
    }
    /* compute mean and variance */
    rho_mean/=num_pareto_points;
    for (i=0; i<evalpts->numcurrpts; i++) {
      if (pareto_optimal[i]) {
        rho_var+=(rho[i]-rho_mean)*(rho[i]-rho_mean);
      }
    }
    rho_var/=num_pareto_points;
    /* ratio-reject cutoff = mean + 3*stddev */
    double cutoff=rho_mean+3*sqrt(rho_var);
    /* now reject the outliers */
    for (i=0; i<evalpts->numcurrpts; i++) {
      if (pareto_optimal[i] && rho[i]>cutoff) {
        printf("Extrapolation: outlier cv=%lf, x=[",penalty[i]);
        if (DIMX) {
          printf("%lf",currpts(evalpts)[i][0]);
        }
        for (j=1; j<(size_t)DIMX; j++) {
          printf(",%lf",currpts(evalpts)[i][j]);
        }
        printf("] rejected (rho=%lf>%lf)\n",rho[i],cutoff);
        pareto_optimal[i]=0;
        num_pareto_points--;
      }
    }
  }
  /* compute the cubic regression and extrapolate to 0 using Cramer's rule */
  double S0=0.,S1=0.,S2=0.,S3=0.,S4=0.,S5=0.,S6=0.;
  double Sx0[DIMX],Sx1[DIMX],Sx2[DIMX],Sx3[DIMX];
  memset(Sx0, 0, sizeof(Sx0));
  memset(Sx1, 0, sizeof(Sx1));
  memset(Sx2, 0, sizeof(Sx2));
  memset(Sx3, 0, sizeof(Sx3));
  for (i=0; i<evalpts->numcurrpts; i++) {
    if (pareto_optimal[i]) {
      double cv=penalty[i];
      double cvn=cv;
      double *x=currpts(evalpts)[i];
      double cvxn[DIMX];
      printf("Extrapolation considering cv=%lf, x=[",cv);
      if (DIMX) {
        printf("%lf",x[0]);
      }
      for (j=1; j<(size_t)DIMX; j++) {
        printf(",%lf",x[j]);
      }
      printf("]\n");
      S0+=1.;
      for (j=0; j<(size_t)DIMX; j++) Sx0[j]+=x[j];
      S1+=cv;
      for (j=0; j<(size_t)DIMX; j++) {
        cvxn[j]=cv*x[j];
        Sx1[j]+=cvxn[j];
      }
      cvn*=cv;
      S2+=cvn;
      for (j=0; j<(size_t)DIMX; j++) {
        cvxn[j]*=cv;
        Sx2[j]+=cvxn[j];
      }
      cvn*=cv;
      S3+=cvn;
      for (j=0; j<(size_t)DIMX; j++) {
        cvxn[j]*=cv;
        Sx3[j]+=cvxn[j];
      }
      cvn*=cv;
      S4+=cvn;
      cvn*=cv;
      S5+=cvn;
      cvn*=cv;
      S6+=cvn;
    }
  }
  double d1=S6*S4*S2+S5*S3*S4+S4*S5*S3-S4*S4*S4-S3*S3*S6-S2*S5*S5;
  double d2=S6*S4*S1+S5*S3*S3+S4*S5*S2-S3*S4*S4-S2*S3*S6-S1*S5*S5;
  double d3=S6*S3*S1+S5*S2*S3+S4*S4*S2-S3*S3*S4-S2*S2*S6-S1*S4*S5;
  double d4=S5*S3*S1+S4*S2*S3+S3*S4*S2-S3*S3*S3-S2*S2*S5-S1*S4*S4;
  double D=S0*d1-S1*d2+S2*d3-S3*d4;
  if (D<DBL_EPSILON*128. && D>-DBL_EPSILON*128.) {
    printf("Extrapolation matrix too ill-conditioned (D=%lg)\n",D);
    return;
  }
  double optimum_x[DIMX];
  for (j=0; j<(size_t)DIMX; j++) {
    optimum_x[j]=(Sx0[j]*d1-Sx1[j]*d2+Sx2[j]*d3-Sx3[j]*d4)/D;
  }
  printf("Extrapolation found x=[");
  if (DIMX) {
    printf("%lf",optimum_x[0]);
  }
  for (i=1; i<(size_t)DIMX; i++) {
    printf(",%lf",optimum_x[i]);
  }
  printf("]\n");
  /* No point in updating equality constraint estimates that will not be used
     anymore, so just pass eqc_estimates = NULL. */
  new_point(evalpts,problem,NULL,optimum_x,evaluate_F,eval_user_data);
}
