/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2012 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#define _ISOC99_SOURCE
#define _XOPEN_SOURCE
#include "gmmem.h"
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include "eval.h"
#include "xmalloc.h"
#include "nlpopt.h"
#include "problem.h"

#ifdef __GNUC__
#define ATTR_UNUSED __attribute__((unused))
#else
#define ATTR_UNUSED /**/
#endif

static const size_t n_steps=10;

struct GMM {
  size_t num_gaussians;
  double *GMM_P;
  double *GMM_L_p;
#define GMM_L(gmm) ((double(*)[DIMX][DIMX])(gmm)->GMM_L_p)
  double *GMM_Xbar_p;
#define GMM_Xbar(gmm) ((double(*)[DIMX])(gmm)->GMM_Xbar_p)
};

/* MX = C^-1 X = (L LT)^-1 X = L^-T L^-1 X */
static void compute_MiX(const struct GMM *gmm,
                        const struct bbowda_problem *problem, const double *X,
                        double *MX, size_t idx)
{
  size_t i,j;
  /* MX := L^-1 X */
  for (i=0; i<(size_t)DIMX; i++) {
    MX[i]=X[i];
    for (j=0; j<i; j++) {
      MX[i]-=GMM_L(gmm)[idx][i][j]*MX[j];
    }
    MX[i]/=GMM_L(gmm)[idx][i][i];
  }
  /* MX := L^-T X */
  /* i is unsigned and will wrap to max size_t after reaching 0 */
  for (i=(size_t)DIMX-1; i<(size_t)DIMX; i--) {
    for (j=i+1; j<(size_t)DIMX; j++) {
      MX[i]-=GMM_L(gmm)[idx][j][i]*MX[j];
    }
    MX[i]/=GMM_L(gmm)[idx][i][i];
  }
}

/* p = exp(-1/2 * (X-Xbar)T M (X-Xbar)) / ((2*pi)^(N/2) * det(L)) */
static double gmm_prob_i(const struct GMM *gmm,
                         const struct bbowda_problem *problem, const double *X,
                         size_t idx)
{
  size_t i;
  double x[DIMX],Mx[DIMX],p=0.;

  /* (X-Xbar)T M (X-Xbar) */
  for (i=0; i<(size_t)DIMX; i++) {
    x[i]=X[i]-GMM_Xbar(gmm)[idx][i];
  }
  compute_MiX(gmm,problem,x,Mx,idx);
  for (i=0; i<(size_t)DIMX; i++) {
    p+=x[i]*Mx[i];
  }

  /* exp(-1/2 * (X-Xbar)T M (X-Xbar)) / ((2*pi)^(N/2)) */
  p=exp((-.5)*p)/pow(2.*M_PI,(double)DIMX*.5);

  /* divide by det(L) */
  for (i=0; i<(size_t)DIMX; i++) {
    p/=GMM_L(gmm)[idx][i][i];
  }

  /* if p is infinite, use our fake infinity instead */
  if (isinf(p)) p=isinf(INF)?1e20:INF;

  return p;
}

double gmm_prob(const struct GMM *gmm, const struct bbowda_problem *problem,
                const double *X)
{
  double p=0.;
  size_t idx;
  for (idx=0; idx<gmm->num_gaussians; idx++) {
    p+=gmm_prob_i(gmm,problem,X,idx)*gmm->GMM_P[idx];
  }
  return p;
}

/* g = - (exp(-1/2 * (X-Xbar)T M (X-Xbar)) * M (X-Xbar)) / ((2*pi)^(N/2) * det(L))
     = -p * M (X-Xbar)
   We have to sum this over all Gaussians. */
void gmm_grad(const struct GMM *gmm, const struct bbowda_problem *problem,
              const double *X, double *g)
{
  size_t idx,i;
  for (i=0; i<(size_t)DIMX; i++) {
    g[i]=0.;
  }

  for (idx=0; idx<gmm->num_gaussians; idx++) {
    double x[DIMX],Mx[DIMX],p;

    /* add -p * M (X-Xbar) to g */
    for (i=0; i<(size_t)DIMX; i++) {
      x[i]=X[i]-GMM_Xbar(gmm)[idx][i];
    }
    compute_MiX(gmm,problem,x,Mx,idx);
    p=gmm_prob_i(gmm,problem,X,idx)*gmm->GMM_P[idx];
    for (i=0; i<(size_t)DIMX; i++) {
      g[i]-=p*Mx[i];
    }
  }
}

#define DIMH ((DIMX*(DIMX+1))>>1)

/* H = -p * M - g (M (X-Xbar))T
     = -p * M + p * (M (X-Xbar)) * (M (X-Xbar))T
     = p * (M (X-Xbar) (M (X-Xbar))T - M)
   We have to sum this over all Gaussians. */
void gmm_hess(const struct GMM *gmm, const struct bbowda_problem *problem,
              const double *X, double *H)
{
  size_t idx,i,j;
  for (i=0; i<(size_t)DIMH; i++) {
    H[i]=0.;
  }

  for (idx=0; idx<gmm->num_gaussians; idx++) {
    double x[DIMX],Mx[DIMX],Mi[DIMX],p;
    double *ptr=H;

    /* compute M (X-Xbar) */
    for (i=0; i<(size_t)DIMX; i++) {
      x[i]=X[i]-GMM_Xbar(gmm)[idx][i];
    }
    compute_MiX(gmm,problem,x,Mx,idx);
    p=gmm_prob_i(gmm,problem,X,idx)*gmm->GMM_P[idx];
    /* compute Hidx */
    for (i=0; i<(size_t)DIMX; i++) {
      for (j=0; j<(size_t)DIMX; j++) {
        x[j]=(j==i)?1.:0.;
      }
      /* Midx is symmetric, so we'll get away with computing columns instead of rows */
      compute_MiX(gmm,problem,x,Mi,idx);
      for (j=0; j<=i; j++) {
        /* Hidx[i][j] = p * ((Midx (X-Xbar))[i] * (Midx (X-Xbar))[j] - Midx[i][j]) */
        *(ptr++)+=p*(Mx[i]*Mx[j]-Mi[j]);
      }
    }
  }
}

struct GMM * build_density_gmm(const struct bbowda_problem *problem,
                               struct eval_points *evalpts)
{
  size_t n,curridx;
  double (*weights)[evalpts->numcurrpts];

  /* Initialize GMM */
  struct GMM *gmm=xmalloc(sizeof(struct GMM));
  gmm->num_gaussians=evalpts->numcurrpts>>2;
  gmm->GMM_P=xmalloc(gmm->num_gaussians*sizeof(double));
  gmm->GMM_L_p=xmalloc(gmm->num_gaussians*(size_t)DIMX*(size_t)DIMX*sizeof(double));
  gmm->GMM_Xbar_p=xmalloc(gmm->num_gaussians*(size_t)DIMX*sizeof(double));
  /* Take every 4th point as starting point for a cluster, rotating points.
     (The same point is used only every 4 iterations.) */
  for (curridx=0, n=((evalpts->numcurrpts-1)&3); curridx<gmm->num_gaussians; curridx++, n+=4) {
    size_t i,j;
    /* Default GMM means to starting points. */
    memcpy(GMM_Xbar(gmm)[curridx],currpts(evalpts)[n],(size_t)DIMX*sizeof(double));
    /* Default GMM priors to 1/gmm->num_gaussians. */
    gmm->GMM_P[curridx]=1./(double)gmm->num_gaussians;
    /* Default GMM covariances to the identity matrix. */
    for (i=0; i<(size_t)DIMX; i++) {
      for (j=0; j<i; j++) {
        GMM_L(gmm)[curridx][i][j]=0.;
      }
      GMM_L(gmm)[curridx][i][i]=1.;
    }
  }

  /* Allocate weight matrix */
  weights=xmalloc(gmm->num_gaussians*evalpts->numcurrpts*sizeof(double));

  /* EM iteration */
  for (n=0; n<n_steps; n++) {
    size_t p;
    /* E Step: compute weights */
    for (p=0; p<evalpts->numcurrpts; p++) {
      /* P(j|x)=p(x|j)*P(j)/p(x), p(x)=sum(j) p(x|j)*P(j) */
      double px=0.;
      for (curridx=0; curridx<gmm->num_gaussians; curridx++) {
        weights[curridx][p]=gmm_prob_i(gmm,problem,currpts(evalpts)[p],curridx)
                            *gmm->GMM_P[curridx];
        px+=weights[curridx][p];
      }
      if (px<DBL_EPSILON) { /* avoid division by zero, default to equiprobability */
        for (curridx=0; curridx<gmm->num_gaussians; curridx++) {
          weights[curridx][p]=1./(double)gmm->num_gaussians;
        }
      } else {
        for (curridx=0; curridx<gmm->num_gaussians; curridx++) {
          weights[curridx][p]/=px;
        }
      }
    }

    /* M Step: compute new covariances */
    for (curridx=0; curridx<gmm->num_gaussians; curridx++) {
      /* compute the covariance matrix */
      size_t i,j,k;
      double n=0.,real_n;
      for (i=0; i<(size_t)DIMX; i++) {
        GMM_Xbar(gmm)[curridx][i]=0.;
      }
      for (p=0; p<evalpts->numcurrpts; p++) {
        n+=weights[curridx][p];
      }
      real_n=n;
      if (!n) {
        n=(double)evalpts->numcurrpts;
        for (p=0; p<evalpts->numcurrpts; p++) {
          weights[curridx][p]=1.;
        }
      }
      for (p=0; p<evalpts->numcurrpts; p++) {
        size_t l;
        for (l=0; l<(size_t)DIMX; l++) {
          GMM_Xbar(gmm)[curridx][l] += weights[curridx][p]*currpts(evalpts)[p][l];
        }
      }
      for (i=0; i<(size_t)DIMX; i++) {
        GMM_Xbar(gmm)[curridx][i]/=n;
      }
      for (i=0; i<(size_t)DIMX; i++) {
        for (j=0; j<=i; j++) {
          GMM_L(gmm)[curridx][i][j]=0.;
        }
      }
      for (p=0; p<evalpts->numcurrpts; p++) {
        size_t l,m;
        for (l=0; l<(size_t)DIMX; l++) {
          for (m=0; m<=l; m++) {
            GMM_L(gmm)[curridx][l][m] += weights[curridx][p]*(currpts(evalpts)[p][l]-GMM_Xbar(gmm)[curridx][l])
                                                       *(currpts(evalpts)[p][m]-GMM_Xbar(gmm)[curridx][m]);
          }
        }
      }
      for (i=0; i<(size_t)DIMX; i++) {
        for (j=0; j<=i; j++) {
          GMM_L(gmm)[curridx][i][j]/=n;
        }
      }

      /* compute the regularized Cholesky factorization of the matrix */
      for (j=0; j<(size_t)DIMX; j++) {
        double Ljj;
        /* regularize */
        if (GMM_L(gmm)[curridx][j][j]<sqrt(DBL_EPSILON)) {
          GMM_L(gmm)[curridx][j][j]=sqrt(DBL_EPSILON);
        }

        Ljj=GMM_L(gmm)[curridx][j][j];
        for (i=j+1; i<(size_t)DIMX; i++) {
          for (k=i; k<(size_t)DIMX; k++) {
            GMM_L(gmm)[curridx][k][i]-=GMM_L(gmm)[curridx][k][j]*GMM_L(gmm)[curridx][i][j]/Ljj;
          }
        }
        for (k=j; k<(size_t)DIMX; k++) {
          GMM_L(gmm)[curridx][k][j]/=sqrt(Ljj);
        }
      }

      /* compute the mixing prior */
      gmm->GMM_P[curridx]=real_n/(double)evalpts->numcurrpts;
    }
  }

  /* free weight matrix */
  free(weights);

  return gmm;
}


void free_gmm(struct GMM *gmm)
{
  free(gmm->GMM_P);
  free(gmm->GMM_L_p);
  free(gmm->GMM_Xbar_p);
  free(gmm);
}
