/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2012 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#define _ISOC99_SOURCE
#include "nlpopt.h"

#define fmin fmin_
#include <math.h>
#undef fmin

#include "donlp2_intv_dyn/o8para.h"
#include "problem.h"
#include "covar.h"
#include "gmmem.h"
#include "eqconst.h"
#include "eval.h"

/* **************************************************************************** */
/*                                 parameters                                   */
/* **************************************************************************** */

#ifdef __GNUC__
#define ATTR_UNUSED __attribute__((unused))
#else
#define ATTR_UNUSED /**/
#endif

static int s_optimization_problem;
static void *s_data;
static int s_ignore_constraints;
static const DOUBLE *s_best_point;
static DOUBLE *s_optimum_x;

/* **************************************************************************** */
/*                              donlp2-intv size initialization                           */
/* **************************************************************************** */
void user_init_size(void) {
    #define  X extern
    #include "o8comm.h"
    #include "o8fint.h"
    #include "o8cons.h"
    #undef   X


    /* problem dimension n = dim(x), nlin=number of linear constraints
       nonlin = number of nonlinear constraints  */

    switch (s_optimization_problem) {
      case 0: {
        const struct bbowda_problem *problem
               = ((struct covar_problem *)s_data)->problem;
        n      = DIMX+DIMZ+DIMY;
        nlin   = 0;
        nonlin = 1+DIMZ;
        break;
      }
      case 1: {
        const struct bbowda_problem *problem
               = ((struct GMM_problem *)s_data)->problem;
        n      = DIMX;
        nlin   = 0;
        nonlin = s_ignore_constraints ? 0 : ((struct GMM_problem *)s_data)
                                            ->estimatec
                                            ->num_estimate_constraints;
        break;
      }
    }
    iterma = 4000;
    nstep = 20;
}


/* **************************************************************************** */
/*                              donlp2-intv standard setup                           */
/* **************************************************************************** */
void user_init(void) {
    #define  X extern
    #include "o8comm.h"
    #include "o8cons.h"
    #undef   X
    
    INTEGER  i,j,k,l;
    DOUBLE optimum=INFINITY;
                                  
    /* name is ident of the example/user and can be set at users will       */
    /* the first static character must be alphabetic. 40 characters maximum */

    strcpy(name,"surrogate");
     
    /* x is initial guess and also holds the current solution */
    /* problem dimension n = dim(x), nlin=number of linear constraints
       nonlin = number of nonlinear constraints  */
    
    analyt = TRUE;
    epsdif = 1.e-16;   /* gradients exact to machine precision */
    /* if you want numerical differentiation being done by donlp2 then:*/
    /* epsfcn   = 1.e-16; *//* function values exact to machine precision */
    /* taubnd   = 5.e-6; */
    /* bounds may be violated at most by taubnd in finite differencing */
    /*  bloc    = TRUE; */
    /* if one wants to evaluate all functions  in an independent process */
    /* difftype = 3; *//* the most accurate and most expensive choice */
    
    nreset = n;
    
    
    del0 = 0.2e0;
    tau0 = 1.e0;
    tau  = 0.1e0;
    big  = INF;
    switch (s_optimization_problem) {
      case 0: {
        const struct bbowda_problem *problem
               = ((struct covar_problem *)s_data)->problem;
        /* starting value: best point */
        for (i = 1 ; i <= DIMX ; i++) {
            x[i] = s_best_point[i-1];
        }
        for (j=1; j<=DIMX; j++) {
            for (k=1; k<=j; k++) {
                x[i++] = x[j]*x[k];
            }
        }
        for (; i <= DIMX+DIMZ+DIMY ; i++) {
            x[i] = s_best_point[i-(DIMZ+1)];
        }

        /*  set lower and upper bounds */
        /* x */
        for (i = 1 ; i <= DIMX ; i++) {
            low[i] = problem->xlow[i-1];
            up[i] = problem->xup[i-1];
        }
        /* z */
        for (j=0; j<DIMX; j++) {
            for (k=0; k<=j; k++) {
                /* interval multiplication [xlow[j],xup[j]]*[xlow[k],xup[k]] */
                DOUBLE bounds[4]={problem->xlow[j]*problem->xlow[k],
                                  problem->xlow[j]*problem->xup[k],
                                  problem->xup[j]*problem->xlow[k],
                                  problem->xup[j]*problem->xup[k]};
                DOUBLE zlow=*bounds, zup=*bounds;
                for (l=1; l<4; l++) {
                    if (bounds[l]<zlow) zlow=bounds[l];
                    if (bounds[l]>zup) zup=bounds[l];
                }
                low[i] = zlow;
                up[i++] = zup;
            }
        }
        /* y */
        for (; i <= DIMX+DIMZ+DIMY ; i++) {
            low[i] = problem->Flow[i-(DIMX+DIMZ+1)];
            up[i] = problem->Fup[i-(DIMX+DIMZ+1)];
        }
        /* covariance model interval constraint */
        low[i] = ((struct covar_problem *)s_data)->covar->klow;
        up[i++] = ((struct covar_problem *)s_data)->covar->kup;
        /* equality constraints */
        for (j=0; j<DIMZ; j++) {
            low[i] = 0.;
            up[i++] = 0.;
        }
        break;
      }
      case 1: {
        const struct bbowda_problem *problem
               = ((struct GMM_problem *)s_data)->problem;
        if (s_ignore_constraints) {
            /* starting value: center of box */
            for (i = 1 ; i <= DIMX ; i++) {
                x[i] = (problem->xlow[i-1] + problem->xup[i-1]) * .5;
            }
        } else {
            /* starting value: unconstrained minimum */
            for (i = 1 ; i <= DIMX ; i++) {
                x[i] = s_optimum_x[i-1];
            }
        }

        /*  set lower and upper bounds */
        for (i = 1 ; i <= DIMX ; i++) {
            low[i] = problem->xlow[i-1];
            up[i] = problem->xup[i-1];
        }
        if (!s_ignore_constraints) {
            /* estimate constraints */
            for (j=0; j<(INTEGER)((struct GMM_problem *)s_data)->estimatec->num_estimate_constraints; j++) {
                low[i] = -((struct GMM_problem *)s_data)->estimate_constraint_tol;
                up[i++] = INF;
            }
        }
        break;
      }
    }

    silent = TRUE;

    return;
}

/* **************************************************************************** */
/*                                 special setup                                */
/* **************************************************************************** */
void setup(void) {
    #define  X extern
    #include "o8comm.h"
    #undef   X
    te0=TRUE;
    /* enforce valid delmin */
    if (!s_optimization_problem
      && delmin + delmin >= ((struct covar_problem *)s_data)->covar->kup
                            - ((struct covar_problem *)s_data)->covar->klow)
      delmin = (((struct covar_problem *)s_data)->covar->kup
                - ((struct covar_problem *)s_data)->covar->klow) * .499;
    return;
}

/* **************************************************************************** */
/*  the user may add additional computations using the computed solution here   */
/* **************************************************************************** */
void solchk(void) {
    #define  X extern
    #include "o8comm.h"
    #undef   X
    #include "o8cons.h"

    INTEGER i;
    const struct bbowda_problem *problem
      =s_optimization_problem?((struct GMM_problem *)s_data)->problem
                             :((struct covar_problem *)s_data)->problem;
    for (i=0; i<DIMX; i++)
        s_optimum_x[i]=x[i+1];

    return;
}

/* **************************************************************************** */
/*                               objective function                             */
/* **************************************************************************** */
void ef(DOUBLE x[],DOUBLE *fx) {
    #define  X extern
    #include "o8fuco.h"
    #undef   X


    INTEGER i;
    DOUBLE f=0.;
    switch (s_optimization_problem) {
      case 0: {
        const struct bbowda_problem *problem
               = ((struct covar_problem *)s_data)->problem;
        for (i=0; i<DIMX; i++) {
            f += problem->c[i]*x[i+1];
        }
        for (i=0; i<DIMY; i++) {
            f += problem->c[i+DIMX]*x[i+(DIMX+DIMZ+1)];
        }
        *fx = f;
        break;
      }
      case 1:
        *fx = gmm_prob(((struct GMM_problem *)s_data)->gmm,
                       ((struct GMM_problem *)s_data)->problem,x+1);
        break;
    }

    return;
}

/* **************************************************************************** */
/*                          gradient of objective function                      */
/* **************************************************************************** */
void egradf(DOUBLE x[],DOUBLE gradf[]) {
    #define  X extern
    #include "o8fuco.h"
    #undef   X

    INTEGER  j;

    switch (s_optimization_problem) {
      case 0: {
        const struct bbowda_problem *problem
               = ((struct covar_problem *)s_data)->problem;
        for (j=0; j<DIMX; j++) {
            gradf[j+1] = problem->c[j];
        }
        for (; j<DIMX+DIMZ; j++) {
            gradf[j+1] = 0.;
        }
        for (j=0; j<DIMY; j++) {
            gradf[j+(DIMX+DIMZ+1)] = problem->c[j+DIMX];
        }
        break;
      }
      case 1:
        gmm_grad(((struct GMM_problem *)s_data)->gmm,
                 ((struct GMM_problem *)s_data)->problem,x+1,gradf+1);
        break;
    }

    return;
}

/* **************************************************************************** */
/*                compute the i-th equality constaint, value is hxi             */
/* **************************************************************************** */
void econ(INTEGER type ,INTEGER liste[], DOUBLE x[],DOUBLE con[], 
              LOGICAL err[]) {
    #define  X extern
    #include "o8fuco.h"
    #undef   X
    const struct bbowda_problem *problem;
    INTEGER i,j,k,l;
    DOUBLE z;
    INTEGER liste_loc_size;
    switch (s_optimization_problem) {
      case 0:
        problem = ((struct covar_problem *)s_data)->problem;
        liste_loc_size = DIMZ+1 ;
        break;
      case 1:
        problem = ((struct GMM_problem *)s_data)->problem;
        liste_loc_size = s_ignore_constraints ? 0 : (INTEGER) ((struct GMM_problem *)s_data)->estimatec->num_estimate_constraints ;
        break;
    }
    INTEGER liste_loc[liste_loc_size+1];
/* if type != 1 only a selection is evaluated the indices being taken from    */
/* liste. since we have no evaluation errors here err is never touched        */


    if ( type == 1 ) 
    {
       liste_loc[0] = liste_loc_size ;
       for ( i = 1 ; i<=liste_loc[0] ; i++ ) { liste_loc[i] = i ; }
    }
    else
    {
       liste_loc[0] = liste[0] ;
       for ( i = 1 ; i<=liste[0] ; i++ ) { liste_loc[i] = liste[i];}
    }
    for ( j = 1 ; j <= liste_loc[0] ; j++ )
    {
      i = liste_loc[j] ;

      switch (s_optimization_problem) {
        case 0:
          if (i==1) {
              /* compute (X-Xbar)T M (X-Xbar) */
              DOUBLE X[DIMX+DIMZ+DIMY+DIMY_EQ], MX[DIMX+DIMZ+DIMY+DIMY_EQ];
              z=0.;
              for (k=0; k<DIMX+DIMZ+DIMY; k++) {
                  X[k]=x[k+1]-((struct covar_problem *)s_data)->covar->Xbar[k];
              }
              for (; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
                  X[k]=-((struct covar_problem *)s_data)->covar->Xbar[k];
              }
              compute_MX(((struct covar_problem *)s_data)->covar,problem,X,MX);
              for (k=0; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
                  z+=X[k]*MX[k];
              }
              con[1]=z;
          } else {
              /* compute zi-xk*xl */
              k=0;
              l=i-2;
              while (l>k) {
                  k++;
                  l-=k;
              }
              con[i]=x[i+(DIMX-1)]-x[k+1]*x[l+1];
          }
          break;
        case 1:
          {
            /* evaluate quadratic polynomial */
            DOUBLE *p=estimate_constraint_coeffs(((struct GMM_problem *)s_data)->estimatec)[i-1];
            z=*(p++); /* constant term */
            for (k=1; k<=DIMX; k++) { /* linear terms */
              z+=*(p++) * x[k];
            }
            for (k=1; k<=DIMX; k++) { /* quadratic/bilinear terms */
              for (l=1; l<=k; l++) {
                z+=*(p++) * (x[k]*x[l]);
              }
            }
            con[i]=z;
          }
          break;
      }

    }
    return;
}

/* **************************************************************************** */
/*              compute the gradient of the i-th equality constraint            */
/* **************************************************************************** */
void econgrad(INTEGER liste[] ,INTEGER shift ,  DOUBLE x[],
               DOUBLE **grad) {
    #define  X extern
    #include "o8fuco.h"
    #undef   X

    const struct bbowda_problem *problem;
    INTEGER  i,j,k,l;
    DOUBLE   z;
    
    INTEGER liste_loc_size;
    switch (s_optimization_problem) {
      case 0:
        problem = ((struct covar_problem *)s_data)->problem;
        liste_loc_size = DIMZ+1 ;
        break;
      case 1:
        problem = ((struct GMM_problem *)s_data)->problem;
        liste_loc_size = s_ignore_constraints ? 0 : (INTEGER) ((struct GMM_problem *)s_data)->estimatec->num_estimate_constraints ;
        break;
    }
    INTEGER liste_loc[liste_loc_size+1];

    liste_loc[0] = liste[0] ;
    for ( i = 1 ; i<=liste_loc[0] ; i++ ) { liste_loc[i] = liste[i];}
    for ( j = 1 ; j <= liste_loc[0] ; j++ )
    {
      i = liste_loc[j] ;
      switch (s_optimization_problem) {
        case 0:
          if (i==1) {
              /* compute ((X-Xbar)T M (X-Xbar))' = 2 M (X-Xbar) */
              DOUBLE X[DIMX+DIMZ+DIMY+DIMY_EQ], MX[DIMX+DIMZ+DIMY+DIMY_EQ];
              for (k=0; k<DIMX+DIMZ+DIMY; k++) {
                  X[k]=x[k+1]-((struct covar_problem *)s_data)->covar->Xbar[k];
              }
              for (; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
                  X[k]=-((struct covar_problem *)s_data)->covar->Xbar[k];
              }
              compute_MX(((struct covar_problem *)s_data)->covar,problem,X,MX);
              for (k=0; k<DIMX+DIMZ+DIMY; k++) {
                  z=MX[k];
                  grad[k+1][i+shift] = 2.*z;
              }
          } else {
              for (k=1; k<=DIMX+DIMZ+DIMY; k++) {
                  grad[k][i+shift] = 0.e0;
              }
              /* compute (zi-xk*xl)' = ezi - xl exk - xk exl */
              k=0;
              l=i-2;
              while (l>k) {
                  k++;
                  l-=k;
              }
              grad[i+(DIMX-1)][i+shift] = 1.;
              grad[k+1][i+shift] -= x[l+1];
              grad[l+1][i+shift] -= x[k+1];
          }
          break;
        case 1:
          {
            /* evaluate gradient of the quadratic polynomial */
            DOUBLE *p=estimate_constraint_coeffs(((struct GMM_problem *)s_data)->estimatec)[i-1]+1; /* skip constant term */
            for (k=1; k<=DIMX; k++) { /* linear terms (constant derivatives) */
              grad[k][i+shift]=*(p++);
            }
            for (k=1; k<=DIMX; k++) { /* quadratic/bilinear terms (linear derivatives) */
              for (l=1; l<=k; l++) {
                grad[k][i+shift]+=*p * x[l];
                grad[l][i+shift]+=*(p++) * x[k];
                /* If k==l, then this is grad[k][i+shift]+=2 * *(p++) * x[k],
                   which is correct, since d(xk^2)/dxk=2 xk.
                   For the bilinear terms, d(xk*xl)/dxk=xl, d(xk*xl)/dxl=xk. */
              }
            }
          }
          break;
      }

    }
    return;
}


/* **************************************************************************** */
/*                        user functions (if bloc == TRUE)                      */
/* **************************************************************************** */
void eval_extern(INTEGER mode) {
    #define  X extern
    #include "o8comm.h"
    #include "o8fint.h"
    #undef   X
    #include "o8cons.h"

    return;
}


/* **************************************************************************** */
/*                        main NLP solver function                              */
/* **************************************************************************** */
void donlp2(void);
extern REAL optite;
int solve_nlp(int optimization_problem, void *data, int ignore_constraints,
              const double *best_point, double *optimum_x) {
    s_optimization_problem = optimization_problem;
    s_data = data;
    s_ignore_constraints = ignore_constraints;
    s_best_point = best_point;
    s_optimum_x = optimum_x;
    donlp2();
    return (int)optite+11;
}
