/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include "donlp3_c_binding.h"

#include "donlp3_c.h"

#if __cplusplus < 201103L
#define override /**/
#endif

struct donlp3_problem : public donlp3_c {
  donlp3_problem(int nx, int nlincons, int nnonlincons, int maxit, int nstep,
                 donlp3_user_init_callback_t user_init_callback,
                 donlp3_setup_callback_t setup_callback,
                 donlp3_solchk_callback_t solchk_callback,
                 donlp3_ef_callback_t ef_callback,
                 donlp3_egradf_callback_t egradf_callback,
                 donlp3_econ_callback_t econ_callback,
                 donlp3_econgrad_callback_t econgrad_callback,
                 donlp3_eval_extern_callback_t eval_extern_callback,
                 void *user_data)
    : donlp3_c(nx, nlincons + nnonlincons, maxit, nstep),
      m_user_init_callback(user_init_callback),
      m_setup_callback(setup_callback),
      m_solchk_callback(solchk_callback),
      m_ef_callback(ef_callback),
      m_egradf_callback(egradf_callback),
      m_econ_callback(econ_callback),
      m_econgrad_callback(econgrad_callback),
      m_eval_extern_callback(eval_extern_callback),
      m_user_data(user_data) {
        n=nx;
        nlin=nlincons;
        nonlin=nnonlincons;
        m_comm.fuco.val = &val;
        m_comm.fuco.llow = &llow;
        m_comm.fuco.lup = &lup;
        m_comm.fuco.n = &n;
        m_comm.fuco.nr = &nr;
        m_comm.fuco.nres = &nres;
        m_comm.fuco.nlin = &nlin;
        m_comm.fuco.nonlin = &nonlin;
        m_comm.fuco.epsmac = &epsmac;
        m_comm.fuco.tolmac = &tolmac;
        m_comm.fuco.deldif = &deldif;
        m_comm.fuco.name = &name;
        m_comm.fuco.epsdif = &epsdif;
        m_comm.fuco.intakt = &intakt;
        m_comm.fuco.te0 = &te0;
        m_comm.fuco.te1 = &te1;
        m_comm.fuco.te2 = &te2;
        m_comm.fuco.te3 = &te3;
        m_comm.fuco.singul = &singul;
        m_comm.fuco.ident = &ident;
        m_comm.fuco.eqres = &eqres;
        m_comm.fuco.silent = &silent;
        m_comm.fuco.analyt = &analyt;
        m_comm.fuco.cold = &cold;
        m_comm.fuco.icf = &icf;
        m_comm.fuco.icgf = &icgf;
        m_comm.fuco.cfincr = &cfincr;
        m_comm.fuco.cres = &cres;
        m_comm.fuco.cgres = &cgres;
        m_comm.fuco.ffuerr = &ffuerr;
        m_comm.fuco.confuerr = &confuerr;
        m_comm.fuco.nh = &nh;
        m_comm.fuco.ng = &ng;
        m_comm.fuco.nonlinlist = &nonlinlist;
        m_comm.fuco.gunit = &gunit;
        m_comm.fuco.gconst = &gconst;
        m_comm.fuco.cfuerr = &cfuerr;
        m_comm.runtim = &runtim;
        m_comm.optite = &optite;
        m_comm.accinf = &accinf;
        m_comm.itstep = &itstep;
        m_comm.phase = &phase;
        m_comm.upsi = &upsi;
        m_comm.upsi0 = &upsi0;
        m_comm.upsi1 = &upsi1;
        m_comm.upsist = &upsist;
        m_comm.psi = &psi;
        m_comm.psi0 = &psi0;
        m_comm.psi1 = &psi1;
        m_comm.psist = &psist;
        m_comm.psimin = &psimin;
        m_comm.phi = &phi;
        m_comm.phi0 = &phi0;
        m_comm.phi1 = &phi1;
        m_comm.phimin = &phimin;
        m_comm.fx = &fx;
        m_comm.fx0 = &fx0;
        m_comm.fx1 = &fx1;
        m_comm.fxst = &fxst;
        m_comm.f_min = &f_min;
        m_comm.b2n = &b2n;
        m_comm.b2n0 = &b2n0;
        m_comm.xnorm = &xnorm;
        m_comm.x0norm = &x0norm;
        m_comm.sig0 = &sig0;
        m_comm.dscal = &dscal;
        m_comm.dnorm = &dnorm;
        m_comm.d0norm = &d0norm;
        m_comm.sig = &sig;
        m_comm.sigmin = &sigmin;
        m_comm.dirder = &dirder;
        m_comm.cosphi = &cosphi;
        m_comm.upsim = &upsim;
        m_comm.x = &x;
        m_comm.x0 = &x0;
        m_comm.x1 = &x1;
        m_comm.xmin = &xmin;
        m_comm.d = &d;
        m_comm.d0 = &d0;
        m_comm.dd = &dd;
        m_comm.difx = &difx;
        m_comm.resmin = &resmin;
        m_comm.gradf = &gradf;
        m_comm.gfn = &gfn;
        m_comm.qgf = &qgf;
        m_comm.gphi0 = &gphi0;
        m_comm.gphi1 = &gphi1;
        m_comm.gres = &gres;
        m_comm.gresn = &gresn;
        m_comm.perm = &perm;
        m_comm.perm1 = &perm1;
        m_comm.colno = &colno;
        m_comm.rank = &rank;
        m_comm.qr = &qr;
        m_comm.betaq = &betaq;
        m_comm.diag = &diag;
        m_comm.cscal = &cscal;
        m_comm.colle = &colle;
        m_comm.a = &a;
        m_comm.scalm = &scalm;
        m_comm.scalm2 = &scalm2;
        m_comm.diag0 = &diag0;
        m_comm.matsc = &matsc;
        m_comm.violis = &violis;
        m_comm.alist = &alist;
        m_comm.bind = &bind;
        m_comm.bind0 = &bind0;
        m_comm.aalist = &aalist;
        m_comm.clist = &clist;
        m_comm.u = &u;
        m_comm.u0 = &u0;
        m_comm.w = &w;
        m_comm.w1 = &w1;
        m_comm.res = &res;
        m_comm.res0 = &res0;
        m_comm.res1 = &res1;
        m_comm.resst = &resst;
        m_comm.scf = &scf;
        m_comm.scf0 = &scf0;
        m_comm.yu = &yu;
        m_comm.slack = &slack;
        m_comm.infeas = &infeas;
        m_comm.work = &work;
        m_comm.iterma = &iterma;
        m_comm.del = &del;
        m_comm.del0 = &del0;
        m_comm.del01 = &del01;
        m_comm.delmin = &delmin;
        m_comm.tau0 = &tau0;
        m_comm.tau = &tau;
        m_comm.ny = &ny;
        m_comm.smalld = &smalld;
        m_comm.smallw = &smallw;
        m_comm.rho = &rho;
        m_comm.rho1 = &rho1;
        m_comm.eta = &eta;
        m_comm.epsx = &epsx;
        m_comm.c1d = &c1d;
        m_comm.scfmax = &scfmax;
        m_comm.updmy0 = &updmy0;
        m_comm.tauqp = &tauqp;
        m_comm.taufac = &taufac;
        m_comm.taumax = &taumax;
        m_comm.alpha = &alpha;
        m_comm.beta = &beta;
        m_comm.theta = &theta;
        m_comm.sigsm = &sigsm;
        m_comm.sigla = &sigla;
        m_comm.delta = &delta;
        m_comm.stptrm = &stptrm;
        m_comm.delta1 = &delta1;
        m_comm.stmaxl = &stmaxl;
        m_comm.level = &level;
        m_comm.clow = &clow;
        m_comm.lastdw = &lastdw;
        m_comm.lastup = &lastup;
        m_comm.lastch = &lastch;
        m_comm.ug = &ug;
        m_comm.og = &og;
        m_comm.low = &low;
        m_comm.up = &up;
        m_comm.big = &big;
        m_comm.nreset = &nreset;
        m_comm.xst = &xst;
        m_comm.prou = &prou;
        m_comm.meu = &meu;
        m_comm.line = &line;
    }
  void update(int nx, int nlincons, int nnonlincons, int maxit, int nstep,
              donlp3_user_init_callback_t user_init_callback,
              donlp3_setup_callback_t setup_callback,
              donlp3_solchk_callback_t solchk_callback,
              donlp3_ef_callback_t ef_callback,
              donlp3_egradf_callback_t egradf_callback,
              donlp3_econ_callback_t econ_callback,
              donlp3_econgrad_callback_t econgrad_callback,
              donlp3_eval_extern_callback_t eval_extern_callback,
              void *user_data) {
    donlp3_c::update(nx, nlincons + nnonlincons, maxit, nstep);
    n=nx;
    nlin=nlincons;
    nonlin=nnonlincons;
    m_user_init_callback = user_init_callback;
    m_setup_callback = setup_callback;
    m_solchk_callback = solchk_callback;
    m_ef_callback = ef_callback;
    m_egradf_callback = egradf_callback;
    m_econ_callback = econ_callback;
    m_econgrad_callback = econgrad_callback;
    m_eval_extern_callback = eval_extern_callback;
    m_user_data = user_data;
  }
  const donlp3_comm *get_comm() const {return &m_comm;}
  void *get_user_data() const {return m_user_data;}
  void set_user_data(void *user_data) {m_user_data = user_data;}
  protected:
    void user_init(void) override;
    void setup(void) override;
    void solchk(void) override;
    void ef(DOUBLE *x, DOUBLE *fx) override;
    void egradf(DOUBLE *x, DOUBLE *gradf) override;
    void econ(INTEGER type, INTEGER *liste, DOUBLE *x, DOUBLE *con,
              LOGICAL *err) override;
    void econgrad(INTEGER *liste, INTEGER shift, DOUBLE *x,
                  DOUBLE **grad) override;
    void eval_extern(INTEGER mode) override;
  private:
    donlp3_comm m_comm;
    donlp3_user_init_callback_t m_user_init_callback;
    donlp3_setup_callback_t m_setup_callback;
    donlp3_solchk_callback_t m_solchk_callback;
    donlp3_ef_callback_t m_ef_callback;
    donlp3_egradf_callback_t m_egradf_callback;
    donlp3_econ_callback_t m_econ_callback;
    donlp3_econgrad_callback_t m_econgrad_callback;
    donlp3_eval_extern_callback_t m_eval_extern_callback;
    void *m_user_data;
};

void donlp3_problem::user_init(void)
{
  if (m_user_init_callback) {
    m_user_init_callback(&m_comm, m_user_data);
  } else {
    donlp3_c::user_init();
  }
}

void donlp3_problem::setup(void)
{
  if (m_setup_callback) {
    m_setup_callback(&m_comm, m_user_data);
  } else {
    donlp3_c::setup();
  }
}

void donlp3_problem::solchk(void)
{
  if (m_solchk_callback) {
    m_solchk_callback(&m_comm, m_user_data);
  } else {
    donlp3_c::solchk();
  }
}

void donlp3_problem::ef(DOUBLE *x, DOUBLE *fx)
{
  if (m_ef_callback) {
    m_ef_callback(x, fx, &(m_comm.fuco), m_user_data);
  } else {
    donlp3_c::ef(x, fx);
  }
}

void donlp3_problem::egradf(DOUBLE *x, DOUBLE *gradf)
{
  if (m_egradf_callback) {
    m_egradf_callback(x, gradf, &(m_comm.fuco), m_user_data);
  } else {
    donlp3_c::egradf(x, gradf);
  }
}

void donlp3_problem::econ(INTEGER type, INTEGER *liste, DOUBLE *x, DOUBLE *con,
                          LOGICAL *err)
{
  if (m_econ_callback) {
    m_econ_callback(type, liste, x, con, err, &(m_comm.fuco), m_user_data);
  } else {
    donlp3_c::econ(type, liste, x, con, err);
  }
}

void donlp3_problem::econgrad(INTEGER *liste, INTEGER shift, DOUBLE *x,
                              DOUBLE **grad)
{
  if (m_econgrad_callback) {
    m_econgrad_callback(liste, shift, x, grad, &(m_comm.fuco), m_user_data);
  } else {
    donlp3_c::econgrad(liste, shift, x, grad);
  }
}

void donlp3_problem::eval_extern(INTEGER mode)
{
  if (m_eval_extern_callback) {
    m_eval_extern_callback(mode, &m_comm, m_user_data);
  } else {
    donlp3_c::eval_extern(mode);
  }
}

struct donlp3_problem *
donlp3_problem_create(int nx, int nlincons, int nnonlincons, int maxit,
                      int nstep, donlp3_user_init_callback_t user_init_callback,
                      donlp3_setup_callback_t setup_callback,
                      donlp3_solchk_callback_t solchk_callback,
                      donlp3_ef_callback_t ef_callback,
                      donlp3_egradf_callback_t egradf_callback,
                      donlp3_econ_callback_t econ_callback,
                      donlp3_econgrad_callback_t econgrad_callback,
                      donlp3_eval_extern_callback_t eval_extern_callback,
                      void *user_data)
{
  return new donlp3_problem(nx, nlincons, nnonlincons, maxit, nstep,
                            user_init_callback, setup_callback, solchk_callback,
                            ef_callback, egradf_callback, econ_callback,
                            econgrad_callback, eval_extern_callback, user_data);
}

void donlp3_problem_update(struct donlp3_problem *donlp3, int nx, int nlincons,
                           int nnonlincons, int maxit, int nstep,
                           donlp3_user_init_callback_t user_init_callback,
                           donlp3_setup_callback_t setup_callback,
                           donlp3_solchk_callback_t solchk_callback,
                           donlp3_ef_callback_t ef_callback,
                           donlp3_egradf_callback_t egradf_callback,
                           donlp3_econ_callback_t econ_callback,
                           donlp3_econgrad_callback_t econgrad_callback,
                           donlp3_eval_extern_callback_t eval_extern_callback,
                           void *user_data)
{
  donlp3->update(nx, nlincons, nnonlincons, maxit, nstep, user_init_callback,
                 setup_callback, solchk_callback, ef_callback, egradf_callback,
                 econ_callback, econgrad_callback, eval_extern_callback,
                 user_data);
}

const struct donlp3_comm * donlp3_problem_get_comm(const struct donlp3_problem
                                                   *donlp3)
{
  return donlp3->get_comm();
}

void * donlp3_problem_get_user_data(const struct donlp3_problem *donlp3)
{
  return donlp3->get_user_data();
}

void donlp3_problem_set_user_data(struct donlp3_problem *donlp3,
                                  void *new_user_data)
{
  donlp3->set_user_data(new_user_data);
}

void donlp3_problem_solve(struct donlp3_problem *donlp3)
{
  donlp3->solve();
}

void donlp3_problem_free(struct donlp3_problem *donlp3)
{
  delete donlp3;
}
