/* bbowda - Black Box Optimization With Data Analysis
   Copyright (C) 2006-2012 Kevin Kofler <Kevin@tigcc.ticalc.org>
   Copyright (C) 2025 DAGOPT Optimization Technologies GmbH (www.dagopt.com)
                      written by Kevin Kofler <kofler@dagopt.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. A copy of the GNU General Public
   License version 3 can be found in the file gpl-3.0.txt.

   Linking bbowda statically or dynamically (directly or indirectly) with
   other modules is making a combined work based on bbowda. Thus, the terms
   and conditions of the GNU General Public License cover the whole
   combination.

   In addition, as a special exception, the copyright holder of bbowda gives
   you permission to combine the bbowda program:
   * with free software programs or libraries that are released under the
     GNU Library or Lesser General Public License (LGPL), either version 2
     of the License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     IBM Common Public License (CPL), either version 1.0 of the License, or
     (at your option) any later version,
   * with free software programs or libraries that are released under the
     eclipse.org Eclipse Public License (EPL), either version 1.0 of the
     License, or (at your option) any later version,
   * with free software programs or libraries that are released under the
     CeCILL-C Free Software License Agreement, either version 1 of the License,
     or (at your option) any later version,
   * with code included in the standard release of MUMPS under the old MUMPS
     Conditions of Use as reproduced in licenses.txt (or modified versions
     of such code, with unchanged license; variants of the license where only
     the list of contributors and/or the list of suggested citations changed
     shall be considered the same license) and
   * if you qualify for a free of charge license of DONLP2, with code
     included in the standard release of DONLP2 under the DONLP2 Conditions
     of Use as reproduced in licenses.txt (or modified versions of such code,
     with unchanged license).
   (For avoidance of doubt, this implies that it is permitted, e.g., to combine
   the bbowda program with current versions of Ipopt released under the EPL
   version 2.0, because 2.0 is >= 1.0. Its dependency MUMPS is released under
   the CeCILL-C version 1, which is also listed above.)

   You may copy and distribute such a system following the terms of the GNU
   GPL for bbowda and the licenses of the other code concerned, provided that
   you include the source code of that other code when and as the GNU GPL
   requires distribution of source code.

   Note that people who make modified versions of bbowda are not obligated
   to grant this special exception for their modified versions; it is their
   choice whether to do so. The GNU General Public License gives permission
   to release a modified version without this exception; this exception also
   makes it possible to release a modified version which carries forward
   this exception.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#define _ISOC99_SOURCE
#include "nlpopt.h"

#include <math.h>
#include <donlp3_c_binding.h>

#include "problem.h"
#include "covar.h"
#include "gmmem.h"
#include "eqconst.h"
#include "eval.h"

/* **************************************************************************** */
/*                                 parameters                                   */
/* **************************************************************************** */

#ifdef __GNUC__
#define ATTR_UNUSED __attribute__((unused))
#else
#define ATTR_UNUSED /**/
#endif

struct nlp_donlp3_data {
  int optimization_problem;
  void *data;
  int ignore_constraints;
  const DOUBLE *best_point;
  DOUBLE *optimum_x;
};

/* **************************************************************************** */
/*                              donlp3 standard setup                           */
/* **************************************************************************** */
static void user_init(const struct donlp3_comm *comm, void *user_data)
{
  INTEGER  i,j,k,l;
  DOUBLE optimum=INFINITY;
                                
  /* name is ident of the example/user and can be set at users will       */
  /* the first static character must be alphabetic. 40 characters maximum */

  strcpy(*(comm->fuco.name),"surrogate");
   
  /* x is initial guess and also holds the current solution */
  /* problem dimension n = dim(x), nlin=number of linear constraints
     nonlin = number of nonlinear constraints  */
  
  *(comm->fuco.analyt)=TRUE;
  *(comm->fuco.epsdif)=1.e-16;   /* gradients exact to machine precision */
  /* if you want numerical differentiation being done by donlp2 then:*/
  /* epsfcn   = 1.e-16; *//* function values exact to machine precision */
  /* taubnd   = 5.e-6; */
  /* bounds may be violated at most by taubnd in finite differencing */
  /*  bloc    = TRUE; */
  /* if one wants to evaluate all functions  in an independent process */
  /* difftype = 3; *//* the most accurate and most expensive choice */
  
  *(comm->nreset)=*(comm->fuco.n);
  
  *(comm->del0)=0.2e0;
  *(comm->tau)=1.e0;
  *(comm->tau)=0.1e0;
  *(comm->big)=INF;
  switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
    case 0: {
      const struct bbowda_problem *problem
             = ((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      /* starting value: best point */
      for (i=1; i<=DIMX; i++) {
        (*(comm->x))[i]=((struct nlp_donlp3_data *)user_data)->best_point[i-1];
      }
      for (j=1; j<=DIMX; j++) {
        for (k=1; k<=j; k++) {
          (*(comm->x))[i++]=(*(comm->x))[j]*(*(comm->x))[k];
        }
      }
      for (; i<=DIMX+DIMZ+DIMY; i++) {
        (*(comm->x))[i]=((struct nlp_donlp3_data *)user_data)
                          ->best_point[i-(DIMZ+1)];
      }

      /*  set lower and upper bounds */
      /* x */
      for (i=1 ; i<=DIMX ; i++) {
        (*(comm->low))[i]=problem->xlow[i-1];
        (*(comm->up))[i]=problem->xup[i-1];
      }
      /* z */
      for (j=0; j<DIMX; j++) {
        for (k=0; k<=j; k++) {
          /* interval multiplication [xlow[j],xup[j]]*[xlow[k],xup[k]] */
          DOUBLE bounds[4]={problem->xlow[j]*problem->xlow[k],
                            problem->xlow[j]*problem->xup[k],
                            problem->xup[j]*problem->xlow[k],
                            problem->xup[j]*problem->xup[k]};
          DOUBLE zlow=*bounds, zup=*bounds;
          for (l=1; l<4; l++) {
            if (bounds[l]<zlow) zlow=bounds[l];
            if (bounds[l]>zup) zup=bounds[l];
          }
          (*(comm->low))[i]=zlow;
          (*(comm->up))[i++]=zup;
        }
      }
      /* y */
      for (; i<=DIMX+DIMZ+DIMY ; i++) {
          (*(comm->low))[i]=problem->Flow[i-(DIMX+DIMZ+1)];
          (*(comm->up))[i]=problem->Fup[i-(DIMX+DIMZ+1)];
      }
      /* covariance model interval constraint */
      (*(comm->low))[i]=((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->klow;
      (*(comm->up))[i++]=((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->kup;
      /* equality constraints */
      for (j=0; j<DIMZ; j++) {
          (*(comm->low))[i]=0.;
          (*(comm->up))[i++]=0.;
      }
      break;
    }
    case 1: {
      const struct bbowda_problem *problem
             =((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      if (((struct nlp_donlp3_data *)user_data)->ignore_constraints) {
        /* starting value: center of box */
        for (i=1; i<=DIMX; i++) {
          (*(comm->x))[i]=(problem->xlow[i-1] + problem->xup[i-1]) * .5;
        }
      } else {
        /* starting value: unconstrained minimum */
        for (i=1; i<=DIMX; i++) {
          (*(comm->x))[i]=((struct nlp_donlp3_data *)user_data)->optimum_x[i-1];
        }
      }

      /*  set lower and upper bounds */
      for (i=1; i<=DIMX; i++) {
        (*(comm->low))[i]=problem->xlow[i-1];
        (*(comm->up))[i]=problem->xup[i-1];
      }
      if (!((struct nlp_donlp3_data *)user_data)->ignore_constraints) {
        /* estimate constraints */
        for (j=0; j<(INTEGER)((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimatec->num_estimate_constraints; j++) {
          (*(comm->low))[i]=-((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimate_constraint_tol;
          (*(comm->up))[i++]=INF;
        }
      }
      break;
    }
  }

  *(comm->fuco.silent)=TRUE;

  return;
}

/* **************************************************************************** */
/*                                 special setup                                */
/* **************************************************************************** */
static void setup(const struct donlp3_comm *comm, void *user_data)
{
  *(comm->fuco.te0)=TRUE;
  /* enforce valid delmin */
  if (!((struct nlp_donlp3_data *)user_data)->optimization_problem
    && *(comm->delmin) + *(comm->delmin) >= ((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->kup
                          - ((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->klow)
    *(comm->delmin)=(((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->kup
              - ((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->klow) * .499;
  return;
}

/* **************************************************************************** */
/*  the user may add additional computations using the computed solution here   */
/* **************************************************************************** */
static void solchk(const struct donlp3_comm *comm, void *user_data)
{
  INTEGER i;
  const struct bbowda_problem *problem
    =((struct nlp_donlp3_data *)user_data)->optimization_problem?((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem
                           :((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
  for (i=0; i<DIMX; i++)
    ((struct nlp_donlp3_data *)user_data)->optimum_x[i]=(*(comm->x))[i+1];

  return;
}

/* **************************************************************************** */
/*                               objective function                             */
/* **************************************************************************** */
static void ef(DOUBLE x[],DOUBLE *fx, const struct donlp3_fuco *fuco,
               void *user_data)
{
  INTEGER i;
  DOUBLE f=0.;
  switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
    case 0: {
      const struct bbowda_problem *problem
             =((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      for (i=0; i<DIMX; i++) {
          f += problem->c[i]*x[i+1];
      }
      for (i=0; i<DIMY; i++) {
          f += problem->c[i+DIMX]*x[i+(DIMX+DIMZ+1)];
      }
      *fx=f;
      break;
    }
    case 1:
      *fx=gmm_prob(((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->gmm,
                     ((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem,x+1);
      break;
  }

  return;
}

/* **************************************************************************** */
/*                          gradient of objective function                      */
/* **************************************************************************** */
static void egradf(DOUBLE x[],DOUBLE gradf[], const struct donlp3_fuco *fuco,
                   void *user_data)
{
  INTEGER  j;

  switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
    case 0: {
      const struct bbowda_problem *problem
             =((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      for (j=0; j<DIMX; j++) {
          gradf[j+1]=problem->c[j];
      }
      for (; j<DIMX+DIMZ; j++) {
          gradf[j+1]=0.;
      }
      for (j=0; j<DIMY; j++) {
          gradf[j+(DIMX+DIMZ+1)]=problem->c[j+DIMX];
      }
      break;
    }
    case 1:
      gmm_grad(((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->gmm,
               ((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem,x+1,gradf+1);
      break;
  }

  return;
}

/* **************************************************************************** */
/*                compute the i-th equality constaint, value is hxi             */
/* **************************************************************************** */
static void econ(INTEGER type ,INTEGER liste[], DOUBLE x[],DOUBLE con[], 
                 LOGICAL err[], const struct donlp3_fuco *fuco, void *user_data)
{
  const struct bbowda_problem *problem;
  INTEGER i,j,k,l;
  DOUBLE z;
  INTEGER liste_loc_size;
  switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
    case 0:
      problem=((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      liste_loc_size=DIMZ+1 ;
      break;
    case 1:
      problem=((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      liste_loc_size=((struct nlp_donlp3_data *)user_data)->ignore_constraints ? 0 : (INTEGER) ((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimatec->num_estimate_constraints ;
      break;
  }
  INTEGER liste_loc[liste_loc_size+1];
/* if type != 1 only a selection is evaluated the indices being taken from    */
/* liste. since we have no evaluation errors here err is never touched        */

  if (type==1) 
  {
   liste_loc[0]=liste_loc_size;
   for (i=1; i<=liste_loc[0]; i++) { liste_loc[i]=i; }
  }
  else
  {
   liste_loc[0]=liste[0];
   for (i=1; i<=liste[0]; i++) { liste_loc[i]=liste[i];}
  }
  for (j=1; j<=liste_loc[0]; j++)
  {
    i=liste_loc[j] ;

    switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
      case 0:
        if (i==1) {
          /* compute (X-Xbar)T M (X-Xbar) */
          DOUBLE X[DIMX+DIMZ+DIMY+DIMY_EQ], MX[DIMX+DIMZ+DIMY+DIMY_EQ];
          z=0.;
          for (k=0; k<DIMX+DIMZ+DIMY; k++) {
            X[k]=x[k+1]-((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->Xbar[k];
          }
          for (; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
            X[k]=-((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->Xbar[k];
          }
          compute_MX(((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar,problem,X,MX);
          for (k=0; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
            z+=X[k]*MX[k];
          }
          con[1]=z;
        } else {
          /* compute zi-xk*xl */
          k=0;
          l=i-2;
          while (l>k) {
            k++;
            l-=k;
          }
          con[i]=x[i+(DIMX-1)]-x[k+1]*x[l+1];
        }
        break;
      case 1: {
          /* evaluate quadratic polynomial */
          DOUBLE *p=estimate_constraint_coeffs(((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimatec)[i-1];
          z=*(p++); /* constant term */
          for (k=1; k<=DIMX; k++) { /* linear terms */
            z+=*(p++) * x[k];
          }
          for (k=1; k<=DIMX; k++) { /* quadratic/bilinear terms */
            for (l=1; l<=k; l++) {
              z+=*(p++) * (x[k]*x[l]);
            }
          }
          con[i]=z;
        }
        break;
    }

  }
  return;
}

/* **************************************************************************** */
/*              compute the gradient of the i-th equality constraint            */
/* **************************************************************************** */
static void econgrad(INTEGER liste[] ,INTEGER shift ,  DOUBLE x[],
                     DOUBLE **grad, const struct donlp3_fuco *fuco,
                     void *user_data)
{
  const struct bbowda_problem *problem;
  INTEGER  i,j,k,l;
  DOUBLE   z;
  
  INTEGER liste_loc_size;
  switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
    case 0:
      problem=((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      liste_loc_size=DIMZ+1 ;
      break;
    case 1:
      problem=((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->problem;
      liste_loc_size=((struct nlp_donlp3_data *)user_data)->ignore_constraints ? 0 : (INTEGER) ((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimatec->num_estimate_constraints ;
      break;
  }
  INTEGER liste_loc[liste_loc_size+1];

  liste_loc[0]=liste[0] ;
  for (i=1; i<=liste_loc[0]; i++) { liste_loc[i]=liste[i];}
  for (j=1; j<=liste_loc[0]; j++)
  {
    i=liste_loc[j] ;
    switch (((struct nlp_donlp3_data *)user_data)->optimization_problem) {
      case 0:
        if (i==1) {
          /* compute ((X-Xbar)T M (X-Xbar))' = 2 M (X-Xbar) */
          DOUBLE X[DIMX+DIMZ+DIMY+DIMY_EQ], MX[DIMX+DIMZ+DIMY+DIMY_EQ];
          for (k=0; k<DIMX+DIMZ+DIMY; k++) {
            X[k]=x[k+1]-((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->Xbar[k];
          }
          for (; k<DIMX+DIMZ+DIMY+DIMY_EQ; k++) {
            X[k]=-((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar->Xbar[k];
          }
          compute_MX(((struct covar_problem *)((struct nlp_donlp3_data *)user_data)->data)->covar,problem,X,MX);
          for (k=0; k<DIMX+DIMZ+DIMY; k++) {
            z=MX[k];
            grad[k+1][i+shift]=2.*z;
          }
        } else {
          for (k=1; k<=DIMX+DIMZ+DIMY; k++) {
            grad[k][i+shift]=0.e0;
          }
          /* compute (zi-xk*xl)' = ezi - xl exk - xk exl */
          k=0;
          l=i-2;
          while (l>k) {
            k++;
            l-=k;
          }
          grad[i+(DIMX-1)][i+shift]=1.;
          grad[k+1][i+shift] -= x[l+1];
          grad[l+1][i+shift] -= x[k+1];
        }
        break;
      case 1:
        {
          /* evaluate gradient of the quadratic polynomial */
          DOUBLE *p=estimate_constraint_coeffs(((struct GMM_problem *)((struct nlp_donlp3_data *)user_data)->data)->estimatec)[i-1]+1; /* skip constant term */
          for (k=1; k<=DIMX; k++) { /* linear terms (constant derivatives) */
            grad[k][i+shift]=*(p++);
          }
          for (k=1; k<=DIMX; k++) { /* quadratic/bilinear terms (linear derivatives) */
            for (l=1; l<=k; l++) {
              grad[k][i+shift]+=*p * x[l];
              grad[l][i+shift]+=*(p++) * x[k];
              /* If k==l, then this is grad[k][i+shift]+=2 * *(p++) * x[k],
                 which is correct, since d(xk^2)/dxk=2 xk.
                 For the bilinear terms, d(xk*xl)/dxk=xl, d(xk*xl)/dxl=xk. */
            }
          }
        }
        break;
    }

  }
  return;
}


/* **************************************************************************** */
/*                        user functions (if bloc == TRUE)                      */
/* **************************************************************************** */
static void eval_extern(INTEGER mode, const struct donlp3_comm *comm,
                        void *user_data) {
  return;
}


/* **************************************************************************** */
/*                        main NLP solver function                              */
/* **************************************************************************** */
int solve_nlp(int optimization_problem, void *data, int ignore_constraints,
              const double *best_point, double *optimum_x) {
  struct nlp_donlp3_data user_data={
    optimization_problem,
    data,
    ignore_constraints,
    best_point,
    optimum_x
  };
  INTEGER x,nlin,nonlin;
  switch (optimization_problem) {
    case 0: {
      const struct bbowda_problem *problem
             =((struct covar_problem *)data)->problem;
      x=DIMX+DIMZ+DIMY;
      nlin=0;
      nonlin=1+DIMZ;
      break;
    }
    case 1: {
      const struct bbowda_problem *problem
             =((struct GMM_problem *)data)->problem;
      x=DIMX;
      nlin=0;
      nonlin=ignore_constraints?0:((struct GMM_problem *)data)->estimatec
                                   ->num_estimate_constraints;
      break;
    }
  }
  struct donlp3_problem *donlp3=donlp3_problem_create(x,nlin,nonlin,4000,20,
                                                      user_init,setup,solchk,
                                                      ef,egradf,econ,
                                                      econgrad,eval_extern,
                                                      &user_data);
  donlp3_problem_solve(donlp3);
  REAL optite=*(donlp3_problem_get_comm(donlp3)->optite);
  donlp3_problem_free(donlp3);
  return (int)optite+11;
}
